/*
-----------------------------------------------------
   Application Name: QNetTraf
          File Name: mainwindow.h
         Start Date: 6/12/2009
       Last Changed: $LastChangedDate: 2010-01-19 18:12:52 +1300 (Tue, 19 Jan 2010) $
           Revision: $Revision: 179 $
          Author(s): David Goodwin
          Copyright: (C) Copyright David Goodwin, 2009, 2010
            License: GNU General Public License
   File Description: Header for Main Window
-----------------------------------------------------
        This file is part of QNetTraf.

    This is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This software is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this software; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

File Notes:
------------------
*/
#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include <QtGui/QMainWindow>
#include <QtGui/QGraphicsView>
#include <QtGui/QHBoxLayout>
#include <QtDebug>
#include <QDateTime>
#include <QPicture>
#include <QSystemTrayIcon>
#include <QCloseEvent>
#include <QTimer>
#include <QTabWidget>
#include <QGroupBox>
#include <QVBoxLayout>
#include <QMessageBox>
#include <QDesktopServices>
#include <QUrl>
#include "host.h"
#include "configuration.h"
#include "graphscene.h"
#include "aboutdialog.h"

namespace Ui
{
    class MainWindowClass;
}

class MainWindow : public QMainWindow
{
    Q_OBJECT

    /** Type of icon to display in system tray and center of graph area */
    enum IconType {
        I_NoActivity   = 0x00000000,
        I_Sending      = 0x00000001,
        I_Recieve      = 0x00000002,
        I_Error        = 0x00000004
    };
    Q_DECLARE_FLAGS(IconTypes, IconType)

public:
    MainWindow(QWidget *parent = 0);
    ~MainWindow();

public slots:
    // --------------- Host Slots ---------------
    /** Notification of activity
      \param in Number of bytes recieved in the last second
      \param out Number of bytes sent in the last second
     */
    void activity(long in, long out);

    /** Connection to the host was lost
     */
    void hostDisconnected();
    /** The host has sent us notification of an ERROR. As it is unable to
      tell us exactly what the error was we will assume invalid interface.
     */
    void hostError();

    // --------------- Graph Slots ---------------
    /** The maximum value in one of the graphs has changed
      \param The new maximum value
     */
    void graphMaxValueChanged(long new_max);

    // --------------- Connection Stuff ---------------
    /** Retry connecting to the host. This is called by a timer.
     */
    void retryConnectToHost();

    // --------------- User Interface stuff ---------------
    /** Shows the configuration dialog. If any settings change they will
      be reapplied as required. This may result in log data being wiped
      or the connection to the host being reestablished.
      */
    void showConfigureDialog();

    /** Shows the about dialog
     */
    void showAboutDialog();

    /** Shows help
     */
    void showHelp();

    /** Exits the program
     */
    void exit();

    /** The user clicked on the message being displayed by the System Tray Icon
     */
    void trayIconMessageClicked();

    /** The user clicked on the System Tray Icon. Restore the main window
      \param reason The reason the system tray icon was activated (doubleclick, trigger, etc)
     */
    void trayIconActivated(QSystemTrayIcon::ActivationReason reason);

    /** Called when the IP Address of the monitored interface changes
      \param new_address The new IP Address of the monitored interface.
     */
    void monitoredIPChanged(QString new_address);

protected:
    /** This hides the main window and cancels the close event
     */
    void closeEvent(QCloseEvent *event);

    /** To move the window when the user has the left mouse button down
      \param event Mouse Event
      */
    void mouseMoveEvent(QMouseEvent *event);

    /** To start moving the window when the user presses the left mouse button
      \param event Mouse Event
      */
    void mousePressEvent(QMouseEvent *event);

    /** To stop moving the window when the user releases the mouse button
      \param event Mouse Event
      */
    void mouseReleaseEvent(QMouseEvent *event);

    /** To show or hide the titlebar, menu bar and statusbar when the user
      double clicks in the window
      \param event Mouse Event
      */
    void mouseDoubleClickEvent(QMouseEvent *event);
private:
    /** Converts the units from Bytes to something more suitable for its size.
      \param size The size to convert. This variable will be updated with the
             new size
      \param units The new units of the supplied size. This should be a new
             empty QString when called.
      \param i Put the i in MB? (eg, KiB, MiB, GiB, etc)
     */
    void convertDataSizeUnits(double * size, QString * units, bool i);

    // --------------- Host connection stuff ---------------
    /** Connects to the host specified by the configuration data
      \returns True if connect succeeded, False otherwise.
     */
    bool connectToHost();

    // --------------- System Tray Icon stuff ---------------
    /** Creates the system tray icon and associated menus
     */
    void createTrayIcon();

    /** Shows the Host Connect Failed message in the system tray
     */
    void showHostConnectFailedMessage();

    /** Shows the Host Reconnected message in the system tray
     */
    void showHostReconnectedMessage();

    /** Creates the tab widget and all other associated widgets.
     */
    void createTabs();

    /** Reconfigures the main window enabling or disabling the tabs
      as required.
     */
    void tabEnabledStateChanged();
    // --------------- Activity Icon stuff ---------------
    /** Gets the specified type of activity icon in the specified scheme
      \param type The type of icon to show
      \param scheme The scheme (theme) to show it in
      \returns The requested icon
     */
    QPixmap getActivityIcon(IconTypes type, QString scheme);

    /** Sets the center icon to the specified type and scheme
      \param type The type of icon to show
      \param scheme The scheme (theme) to show it in
     */
    void setCenterIcon(MainWindow::IconTypes type, QString scheme);
    /** Sets the System Tray icon to the specified type and scheme
      \param type The type of icon to show
      \param scheme The scheme (theme) to show it in
     */
    void setTrayIcon(MainWindow::IconTypes type, QString scheme);
    /** Sets the all Activity icons to the specified type and scheme
      \param type The type of icon to show
      \param scheme The scheme (theme) to show it in
     */
    void setActivityIcons(MainWindow::IconTypes type, QString scheme);

    // --------------- GUI Member Variables ---------------
    Ui::MainWindowClass *ui;   /*!< Main Window UI */

    QLabel lblIncomingLabel; /*!< Recieved data label in status bar */
    QLabel lblOutgoingLabel; /*!< Sent data label in status bar */
    QLabel whenLogStarted;   /*!< Date/time when log was started */

    QLabel *lblSent;         /*!< The label above the sent graph */
    QLabel *lblRecvd;        /*!< The label above the Recieved graph */
    QLabel *lblIcon;         /*!< The activity icon between the two graphs */

    GraphScene *in_graph;    /*!< Incoming data graph */
    GraphScene *out_graph;   /*!< Outgoing data graph */

    QGridLayout *layout;    /*!< The layout on the main window */
    QWidget *centralWidget;  /*!< The central widget on the main window */

    // System Tray Icon stuff
    QAction *restoreAction;  /*!< Restore menu item for system tray icon */
    QAction *quitAction;     /*!< Quit menu item for system tray icon */

    QSystemTrayIcon *trayIcon;  /*!< System tray icon */
    QMenu *trayIconMenu;     /*!< System tray icon menu */

    // Tab Widget stuff
    QWidget *generalWidget;
    QTabWidget *tabWidget;
    QGridLayout *glConnectionLayout;
    QGridLayout *glActivityLayout;
    QVBoxLayout *vblGeneralLayout;
    QGroupBox *gbConnection;
    QGroupBox *gbActivity;
    QLabel *lblGStatusLabel;
    QLabel *lblGStatus;
    QLabel *lblGIPAddrLabel;
    QLabel *lblGIpAddr;
    QLabel *lblGIFaceLabel;
    QLabel *lblGIFace;
    QLabel *lblGOutLabel;
    QLabel *lblGOut;
    QLabel *lblGInLabel;
    QLabel *lblGIn;
    bool tabs_created;

    // For click-draging to move window
    bool m_window_moving;    /*!< If true, the window will follow the mouse */
    QPoint dragMovePosition; /*!< Initial position when click-moving */

    // Control visibility of titlebar, etc
    Qt::WindowFlags original_window_flags; /*!< So the titlebar can be put back */
    bool frame_visible;     /*!< If the titlebar, menu, statusbar is visible or not */

    // --------------- Connection State Variables ---------------
    Host *h;    /*!< Host Connection */

    bool drop_sample;   /*!< For dropping the first sample (which is always corrupt) */

    long log_recvd_data; /*!< Total Data Recieved */
    long log_sent_data;  /*!< Total Data Sent */
    QString log_since;  /*!< When logging was started */

    QTimer *reconnect_timer;    /*!< Handles automatic reconnections */


    // --------------- Message state variables ---------------
    bool hostErrorShown;    /*!< So the host connect error only appears once */
    bool hostActivityRecieved;  /*!< To control the interface error message */
    bool failedConnectShown;    /*!< So the failed connection error only appears once */
    bool userReconnecting;  /*!< User has initiated a reconnect so dont spam with messages */
    bool notifyOnActivity; /*!< Notifies the user when there is activity again */
    QString current_url;    /*!< URL to show on message click */
};

#endif // MAINWINDOW_H
