/*
-----------------------------------------------------
   Application Name: QNetTraf
          File Name: mainwindow.cpp
         Start Date: 6/12/2009
       Last Changed: $LastChangedDate: 2010-01-19 18:29:17 +1300 (Tue, 19 Jan 2010) $
           Revision: $Revision: 180 $
          Author(s): David Goodwin
          Copyright: (C) Copyright David Goodwin, 2009, 2010
            License: GNU General Public License
   File Description: Main Window Implementation
-----------------------------------------------------
        This file is part of QNetTraf.

    This is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This software is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this software; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

File Notes:
------------------
*/
#include "mainwindow.h"
#include "ui_mainwindow.h"

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent), ui(new Ui::MainWindowClass)
{
    qDebug() << "MainWindow::MainWindow()";

    // Setup the Main Window
    ui->setupUi(this);
    connect(ui->action_Configuration, SIGNAL(triggered()),this,SLOT(showConfigureDialog()));
    connect(ui->actionE_xit,SIGNAL(triggered()),this,SLOT(exit()));
    connect(ui->action_Show_Help,SIGNAL(triggered()),this,SLOT(showHelp()));
    connect(ui->action_About,SIGNAL(triggered()),this,SLOT(showAboutDialog()));

    ui->action_Show_Help->setChecked(SettingsManager::isHelpEnabled());

    setWindowTitle(QCoreApplication::applicationName());

    // Frame visibility and window moving stuff
    original_window_flags = windowFlags();
    frame_visible = true;
    m_window_moving = false;

    // Reconnect timer
    reconnect_timer = new QTimer();
    reconnect_timer->setSingleShot(true);
    connect(reconnect_timer,SIGNAL(timeout()),this,SLOT(retryConnectToHost()));

    // Tray Icon
    createTrayIcon();
    connect(trayIcon, SIGNAL(messageClicked()), this, SLOT(trayIconMessageClicked()));
    connect(trayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
            this, SLOT(trayIconActivated(QSystemTrayIcon::ActivationReason)));
    trayIcon->show();
    hostErrorShown = false;

    tabs_created = false;

    notifyOnActivity = false;

    // Setup the Tab Widget (if enabled)
    if (!SettingsManager::isTabWidgetEnabled()) {
        generalWidget = 0;
        tabWidget = 0;
        glConnectionLayout = 0;
        glActivityLayout = 0;
        gbConnection = 0;
        gbActivity = 0;
        vblGeneralLayout = 0;
        lblGStatusLabel = 0;
        lblGStatus = 0;
        lblGIPAddrLabel = 0;
        lblGIpAddr = 0;
        lblGIFaceLabel = 0;
        lblGIFace = 0;
        lblGOutLabel = 0;
        lblGOut = 0;
        lblGInLabel = 0;
        lblGIn = 0;
    } else {
        createTabs();
    }

    // Setup the graphs

    qDebug() << "Setting up graph widgets...";

    centralWidget = new QWidget(this);

    layout = new QGridLayout(centralWidget);

    in_graph = new GraphScene(centralWidget);
    out_graph = new GraphScene(centralWidget);

    in_graph->setHelpEnabled(SettingsManager::isHelpEnabled());
    out_graph->setHelpEnabled(SettingsManager::isHelpEnabled());

    connect(in_graph,SIGNAL(maximum_value_changed(long)),this,SLOT(graphMaxValueChanged(long)));
    connect(out_graph,SIGNAL(maximum_value_changed(long)),this,SLOT(graphMaxValueChanged(long)));

    in_graph->setPen(QPen(QColor(0,0,100)));
    out_graph->setPen(QPen(QColor(100,0,0)));

    // outgoing is Left, incoming is Right
    lblSent = new QLabel();
    lblRecvd = new QLabel();
    lblIcon = new QLabel();
    lblSent->setText("Sent:");
    lblRecvd->setText("Received:");
    setCenterIcon(I_NoActivity,SettingsManager::getStatusIconStyle());
    lblIcon->setMaximumSize(32,32);

    layout->addWidget(lblSent, 0, 0, 1, 1);
    layout->addWidget(out_graph, 1, 0, 1, 1);
    layout->addWidget(lblIcon, 1, 1, 1, 1);
    layout->addWidget(in_graph, 1, 2, 1, 1);
    layout->addWidget(lblRecvd, 0, 2, 1, 1);

    centralWidget->setLayout(layout);

    // Reconfigure the UI for either tabs or no tabs - what ever the
    // settings file says
    tabEnabledStateChanged();

    // Setup status bar
    lblOutgoingLabel.setText("Sent: ");
    lblIncomingLabel.setText("Received: ");
    ui->statusBar->addPermanentWidget(&lblOutgoingLabel);
    ui->statusBar->addPermanentWidget(&lblIncomingLabel);
    ui->statusBar->addPermanentWidget(&whenLogStarted);

    // Drop the first sample - it usually seems to be wrong.
    drop_sample = true;

    log_recvd_data = 0;
    log_sent_data = 0;

    // Connect:
    h = 0;

    if (SettingsManager::showConfigDialog()) {
        showConfigureDialog();
    }

    if (h == 0) {
        if (connectToHost())
            setWindowTitle(
                    QCoreApplication::applicationName() +
                    " - " + SettingsManager::getInterfaceName());
        else {
            showHostConnectFailedMessage();
            reconnect_timer->start(SettingsManager::getConnectionRetryTimeout());
        }
    }
}

MainWindow::~MainWindow()
{
    qDebug() << "MainWindow::~MainWindow()";
    delete ui;
    delete h;
    delete in_graph;
    delete out_graph;
    delete lblSent;
    delete lblRecvd;
    delete lblIcon;
    delete layout;
    delete centralWidget;
    delete restoreAction;
    delete quitAction;
    delete trayIconMenu;
    delete trayIcon;
    delete reconnect_timer;

    if (tabWidget != 0) {delete tabWidget; tabWidget = 0;}
}

void MainWindow::activity(long in, long out) {
    qDebug() << "MainWindow::activity(in,out: " + QString::number(in) + "," + QString::number(out) + ")";
    // We received data!!!!
    qDebug() << "IN: " << in << "OUT: " << out;

    if (notifyOnActivity && trayIcon != 0) {
        trayIcon->showMessage(SettingsManager::getNowValidMessageTitle(),
                              SettingsManager::getNowValidMessageBody(),
                              QSystemTrayIcon::Information,
                              SettingsManager::getSysTrayMessageTimeout());
        current_url = SettingsManager::getNowValidMessageURL();
        notifyOnActivity = false;
    }


    MainWindow::IconTypes type;
    if (in > 0)
        type |= MainWindow::I_Recieve;
    if (out > 0)
        type |= MainWindow::I_Sending;

    setActivityIcons(type,SettingsManager::getStatusIconStyle());

    // Compute the window title and tool-tip text
    double *in_value,*out_value;
    in_value = new double;
    out_value = new double;
    QString *in_units, *out_units;
    in_units = new QString();
    out_units = new QString();

    *in_value = (double)in;
    *out_value = (double)out;

    convertDataSizeUnits(in_value,in_units,false);
    convertDataSizeUnits(out_value,out_units,false);

    QString tittip;
    QString format_string = QCoreApplication::applicationName() + " - %s - Out: %.2f %s, In: %.2f %s";
    tittip.sprintf(format_string.toAscii().constData(),SettingsManager::getInterfaceName().toAscii().constData(), *out_value, out_units->toAscii().constData(), *in_value, in_units->toAscii().constData());
    trayIcon->setToolTip(tittip);

    // If the tab widget is enabled, update it
    if (lblGIn != 0 && lblGOut != 0) {
        qDebug() << "Tab Widget Enabled - setting activity values...";
        double *in_avg_value = new double;
        double *out_avg_value = new double;
        qDebug() << "Getting averages from graphs...";
        *in_avg_value = in_graph->getAverage();
        *out_avg_value = out_graph->getAverage();

        QString *in_avg_units = new QString();
        QString *out_avg_units = new QString();

        qDebug() << "Converting units...";
        convertDataSizeUnits(in_avg_value,in_avg_units,false);
        convertDataSizeUnits(out_avg_value,out_avg_units,false);

        QString tab_format_string = "%.2f %s (~%.2f %s)";

        qDebug() << "Formatting strings...";
        QString in_string,out_string;
        in_string.sprintf(tab_format_string.toAscii().constData(),
                          *in_value,
                          in_units->toAscii().constData(),
                          *in_avg_value,
                          in_avg_units->toAscii().constData());
        out_string.sprintf(tab_format_string.toAscii().constData(),
                          *out_value,
                          out_units->toAscii().constData(),
                          *out_avg_value,
                          out_avg_units->toAscii().constData());
        qDebug() << "Setting text";
        lblGIn->setText(in_string);
        lblGOut->setText(out_string);

        qDebug() << "Cleaning up...";
        delete in_avg_value;
        delete out_avg_value;
        delete in_avg_units;
        delete out_avg_units;
    }

    delete in_value;
    delete out_value;
    delete in_units;
    delete out_units;

    // Update the graphs if we have been told not to drop the sample.
    if (drop_sample)
        drop_sample = false;
    else {
        hostActivityRecieved = true;



        // Send the samples to the graphs
        qDebug() << "Sending Sample to Graphs...";
        in_graph->addSample(in);
        out_graph->addSample(out);

        qDebug() << "Updating titlebar and statusbar...";
        // Update the title bar if we are supposed to
        if (SettingsManager::isTitlebarSpeedEnabled()) {
            setWindowTitle(tittip);
        }

        // Log data if we are supposed to
        if (SettingsManager::isLogDataEnabled()) {

            if (log_since.isEmpty()) {
                log_since = "Since: " + QDateTime::currentDateTime().toString();
                whenLogStarted.setText(log_since);
            }

            log_recvd_data += in;
            log_sent_data += out;

            double *recvd_data,*sent_data;
            recvd_data = new double;
            sent_data = new double;
            QString *recvd_units, *sent_units;
            recvd_units = new QString();
            sent_units = new QString();

            *recvd_data = (double)log_recvd_data;
            *sent_data = (double)log_sent_data;

            convertDataSizeUnits(recvd_data,recvd_units,false);
            convertDataSizeUnits(sent_data,sent_units,false);

            QString fmt = "Recvd: %.2f %s";
            QString value;
            value.sprintf(fmt.toAscii().constData(), *recvd_data, recvd_units->toAscii().constData());
            lblIncomingLabel.setText(value);

            fmt = "Sent: %.2f %s";
            value = "";
            value.sprintf(fmt.toAscii().constData(),*sent_data, sent_units->toAscii().constData());
            lblOutgoingLabel.setText(value);

            delete recvd_data;
            delete sent_data;
            delete recvd_units;
            delete sent_units;
        }
    }

    // Here is where we should update the system tray icon and middle indicator thing
}

void MainWindow::graphMaxValueChanged(long new_max) {

    // Ok. one of the graphs maximum value has changed. We can't really rely
    // on this value for syncing the two graphs though. So we will ask each
    // graph for its maximum value and give the smaller graph the max value
    // from the larger one. If that makes any sense. Just read the code!

    long in_max = in_graph->get_maximum_value();
    long out_max = out_graph->get_maximum_value();

    qDebug() << ">>>>>>>>>in max: " << in_max << " <<<<<<<<outmax: " << out_max;

    if (in_max > out_max){
        qDebug() << ":::::::::::::::::::::::::::in_graph  has maxvalue - telling out_graph...";
        out_graph->set_maximum_value(in_max);
    }
    else {
        qDebug() << ":::::::::::::::::::::::::::out_graph has maxvalue - telling  in_graph...";
        in_graph->set_maximum_value(out_max);
    }
}

void MainWindow::showConfigureDialog() {
    qDebug() << "show_configure()";

    QString old_hostname, new_hostname;
    int old_port, new_port;
    QString old_interface, new_interface;
    int old_protocolversion, new_protocolversion;
    QString old_password, new_password;
    bool old_logenabled, new_logenabled;
    bool old_titlebarspeed, new_titlebarspeed;
    bool reset_log = false;
    bool old_tabsenabled, new_tabsenabled;

    // First up: Grab the current settings
    old_hostname = SettingsManager::getHostName();
    old_port = SettingsManager::getPort();
    old_interface = SettingsManager::getInterfaceName();
    old_protocolversion = SettingsManager::getProtocolVersion();
    old_password = SettingsManager::getPassword();
    old_logenabled = SettingsManager::isLogDataEnabled();
    old_titlebarspeed = SettingsManager::isTitlebarSpeedEnabled();
    old_tabsenabled = SettingsManager::isTabWidgetEnabled();

    // Show the dialog
    Configuration config_dialog;
    config_dialog.exec();
    SettingsManager::setShowConfigDialog(false);

    // Grab the new settings
    new_hostname = SettingsManager::getHostName();
    new_port = SettingsManager::getPort();
    new_interface = SettingsManager::getInterfaceName();
    new_protocolversion = SettingsManager::getProtocolVersion();
    new_password = SettingsManager::getPassword();
    new_logenabled = SettingsManager::isLogDataEnabled();
    new_titlebarspeed = SettingsManager::isTitlebarSpeedEnabled();
    new_tabsenabled = SettingsManager::isTabWidgetEnabled();

    // Compare before and after to see if we need to make any changes.
    // If any of the following settings have changed we must reconnect:
    // hostname, port, interface, protocolversion, password
    if (old_hostname.compare(new_hostname) != 0 ||
        old_port != new_port ||
        old_interface.compare(new_interface) != 0 ||
        old_protocolversion != new_protocolversion ||
        old_password.compare(new_password) != 0) {
        // A connection setting has changed. We must reconnect.
        userReconnecting = true;
        if (!connectToHost()) {
            userReconnecting = false;
            showHostConnectFailedMessage();
            reconnect_timer->start(SettingsManager::getConnectionRetryTimeout());
        }
        userReconnecting = false;

        reset_log = true;
    }

    // if logenabled has changed we must see if we have to reset logging.
    if (old_logenabled != new_logenabled || reset_log) {
        if (new_logenabled) {
            // Logging has just been enabled. Reset it.
            log_recvd_data = 0;
            log_sent_data = 0;
            log_since = "";
        } else {
            // Logging has just been disabled. Clear it.
            log_recvd_data = 0;
            log_sent_data = 0;
            log_since = "Log Disabled";
            lblOutgoingLabel.setText("");
            lblIncomingLabel.setText("");
            whenLogStarted.setText(log_since);
        }
    }

    // if titlebarspeed has changed we must enable or disable it
    if (old_titlebarspeed != new_titlebarspeed) {
        if (new_titlebarspeed) {
            // Its just been enabled. We don't need to do anything much.
        } else {
            // Its just been disabled
            setWindowTitle(QCoreApplication::applicationName() + " - " + SettingsManager::getInterfaceName());
        }
    }

    // Do we need to reconfigure the GUI?
    if (old_tabsenabled != new_tabsenabled) {
        // yes, we do. We have to restart to do that.
        QMessageBox::information(this,"Restart Required","You must restart RNetTraf for your changes to take effect.");
    }
}

void MainWindow::exit() {
    qDebug() << "exit()";
    h->disconnect();
    QApplication::exit(0);
}

void MainWindow::showHelp() {
    qDebug() << "show_help()";
    SettingsManager::setHelpEnabled(ui->action_Show_Help->isChecked());
    in_graph->setHelpEnabled(ui->action_Show_Help->isChecked());
    out_graph->setHelpEnabled(ui->action_Show_Help->isChecked());
}

void MainWindow::showAboutDialog() {
    qDebug() << "show_about()";
    AboutDialog about_dialog;
    about_dialog.exec();
}

void MainWindow::convertDataSizeUnits(double * size, QString * units, bool i) {
    double avg_speed = *size;
    QString unit = "B";
    if (avg_speed > 1024) {
        avg_speed /= 1024;
        if (i) unit = "KiB"; else unit = "KB";
    }
    if (avg_speed > 1024) {
        avg_speed /= 1024;
        if (i) unit = "MiB"; else unit = "MB";
    }
    if (avg_speed > 1024) {
        avg_speed /= 1024;
        if (i) unit = "GiB"; else unit = "GB";
    }
    if (avg_speed > 1024) {
        avg_speed /= 1024;
        if (i) unit = "TiB"; else unit = "TB";
    }
    *size = avg_speed;
    *units = unit;
    qDebug() << "Units: " << *units;
}

bool MainWindow::connectToHost() {
    qDebug() << "Connecting...";

    if (h != 0) {
        qDebug() << "Already connected. Disconnecting...";
        h->hostDisconnect();
        delete h;
        h = 0;
        drop_sample = true;
        hostErrorShown = false;
        in_graph->reset();
        out_graph->reset();
    }
    hostActivityRecieved = false;

    if (lblGIFace != 0) lblGIFace->setText(SettingsManager::getInterfaceName());

    // Get some data!
    h = new Host(SettingsManager::getHostName(),
                 SettingsManager::getPort(),
                 SettingsManager::getPassword(),
                 SettingsManager::getInterfaceName(),
                 500,
                 SettingsManager::getProtocolVersion() == 1 ? false : true);

    connect(h,SIGNAL(activity(long,long)),this,SLOT(activity(long,long)));
    connect(h,SIGNAL(disconnected()),this,SLOT(hostDisconnected()));
    connect(h,SIGNAL(error()),this,SLOT(hostError()));
    connect(h,SIGNAL(ipset(QString)),this,SLOT(monitoredIPChanged(QString)));

    bool result = h->hostConnect();

    if (result && lblGStatus != 0) lblGStatus->setText("Connected!");

    setWindowTitle(QApplication::applicationName() + " - " + SettingsManager::getInterfaceName());

    return result;
}

QPixmap MainWindow::getActivityIcon(MainWindow::IconTypes type, QString scheme) {
    QString filename = ":/images/activity/" + scheme + "/";

    if (type.testFlag(MainWindow::I_Error)) {
        filename += "ee";
    } else {
        if (type.testFlag(MainWindow::I_Sending))
            filename += "s";
        else
            filename += "x";

        if (type.testFlag(MainWindow::I_Recieve))
            filename += "r";
        else
            filename += "x";
    }
    qDebug() << "Icon Filename: " << filename;
    return QPixmap(filename);
}

void MainWindow::setCenterIcon(MainWindow::IconTypes type, QString scheme) {
    lblIcon->setPixmap(getActivityIcon(type,scheme));
}

void MainWindow::setTrayIcon(MainWindow::IconTypes type, QString scheme) {
    trayIcon->setIcon(QIcon(getActivityIcon(type,scheme)));
}

void MainWindow::setActivityIcons(MainWindow::IconTypes type, QString scheme) {
    setTrayIcon(type,scheme);
    setCenterIcon(type,scheme);
}

void MainWindow::createTrayIcon()
{
    restoreAction = new QAction(tr("&Restore"), this);
    connect(restoreAction, SIGNAL(triggered()), this, SLOT(showNormal()));

    quitAction = new QAction(tr("E&xit"), this);
    connect(quitAction, SIGNAL(triggered()), this, SLOT(exit()));

    trayIconMenu = new QMenu(this);
    trayIconMenu->addAction(restoreAction);
    trayIconMenu->addSeparator();
    trayIconMenu->addAction(quitAction);

    trayIcon = new QSystemTrayIcon(this);
    setTrayIcon(MainWindow::I_NoActivity,SettingsManager::getStatusIconStyle());
    trayIcon->setContextMenu(trayIconMenu);
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    if (trayIcon->isVisible()) {        
        if (SettingsManager::isMinimiseOnCloseEnabled()) {
            if (!SettingsManager::isSystemTrayHintShown()) {
                QMessageBox::information(this, "RNetTraf",
                                     "RNetTraf will keep running in the "
                                        "system tray. To restore it, click on the "
                                        "icon. To exit, choose <b>Exit</b> from "
                                        "the context menu or from the main windows "
                                        "File menu.");
                SettingsManager::setSystemTrayHintShown(true);
            }
            hide();
            event->ignore();
        } else {
            exit();
        }
    }
}

void MainWindow::trayIconActivated(QSystemTrayIcon::ActivationReason reason)
{
    switch (reason) {
    case QSystemTrayIcon::Trigger:
    case QSystemTrayIcon::MiddleClick:
    case QSystemTrayIcon::DoubleClick:
        showNormal();
    default:
        ;
    }
}

void MainWindow::trayIconMessageClicked()
{
    if (current_url.isEmpty())
        showNormal();
    else {
        QDesktopServices::openUrl(QUrl(current_url));
    }
}

void MainWindow::hostError() {
    if (!hostErrorShown) {
        hostErrorShown = true;

        QString message;

        message = SettingsManager::getInvalidInterfaceMessageBody(!hostActivityRecieved);

        trayIcon->showMessage("Error - Invalid Interface '" +
                              SettingsManager::getInterfaceName() + "'",
                              message, QSystemTrayIcon::Warning,
                              SettingsManager::getSysTrayMessageTimeout());
        current_url = SettingsManager::getInvalidInterfaceMessageURL(!hostActivityRecieved);

    }
    setActivityIcons(MainWindow::I_Error, SettingsManager::getStatusIconStyle());

    if (lblGStatus != 0) lblGStatus->setText("Error - Invalid Interface '"
                                             + SettingsManager::getInterfaceName()
                                             + "'");

    notifyOnActivity = true;
}

void MainWindow::hostDisconnected() {
    // Ignore the disconnect if the user is initiating a reconnect.
    // If we don't we will end up in an infinite loop.
    if (userReconnecting == true) return;

    if (lblGStatus != 0) lblGStatus->setText("Connection Lost. Waiting to Reconnect...");

    trayIcon->showMessage(SettingsManager::getDisconnectedMessageTitle(),
                          SettingsManager::getDisconnectedMessageBody(),
                          QSystemTrayIcon::Critical,
                          SettingsManager::getSysTrayMessageTimeout());
    current_url = SettingsManager::getDisconnectedMessageURL();
    setActivityIcons(MainWindow::I_Error,SettingsManager::getStatusIconStyle());
    reconnect_timer->start(SettingsManager::getConnectionRetryTimeout());
}

void MainWindow::showHostConnectFailedMessage() {
    if (!failedConnectShown) {
        trayIcon->showMessage(SettingsManager::getFailedConnectMessageTitle(),
                              SettingsManager::getFailedConnectMessageBody(),
                              QSystemTrayIcon::Critical,
                              SettingsManager::getSysTrayMessageTimeout());
        current_url = SettingsManager::getFailedConnectMessageURL();
        failedConnectShown = true;
    }

}

void MainWindow::showHostReconnectedMessage() {
    trayIcon->showMessage(SettingsManager::getReconnectedMessageTitle(),
                          SettingsManager::getReconnectedMessageBody(),
                          QSystemTrayIcon::Information,
                          SettingsManager::getSysTrayMessageTimeout());
    current_url = SettingsManager::getReconnectedMessageURL();
}

void MainWindow::retryConnectToHost() {
    if (userReconnecting) return;

    if (connectToHost()) {
        if (lblGStatus != 0) lblGStatus->setText("Automatic Reconnect");
        showHostReconnectedMessage();
    } else {
        if (lblGStatus != 0) lblGStatus->setText("Reconnect failed. Waiting to Retry...");
        reconnect_timer->start(SettingsManager::getConnectionRetryTimeout());
    }
}

void MainWindow::mouseMoveEvent(QMouseEvent *event) {
    if (event->buttons().testFlag(Qt::LeftButton) && m_window_moving) {
        move(event->globalPos() - dragMovePosition);
        event->accept();
    }
}

void MainWindow::mousePressEvent(QMouseEvent *event) {
    if (event->button() == Qt::LeftButton) {
        dragMovePosition = event->globalPos() - frameGeometry().topLeft();
        m_window_moving = true;
        event->accept();
    }
}

void MainWindow::mouseReleaseEvent(QMouseEvent *event) {
    if (event->button() == Qt::LeftButton) {
            m_window_moving = false;
    }
}

void MainWindow::mouseDoubleClickEvent(QMouseEvent *event) {
    QPoint orig_pos = pos();
    if (frame_visible) {
        setWindowFlags(Qt::CustomizeWindowHint);
        QPoint orig_pos = pos();
        menuBar()->setVisible(false);
        statusBar()->setVisible(false);
        frame_visible = false;

    } else {
        setWindowFlags(original_window_flags);
        menuBar()->setVisible(true);
        statusBar()->setVisible(true);
        frame_visible = true;
    }
    move(orig_pos);
    show();
}

void MainWindow::monitoredIPChanged(QString new_address) {
    if (lblGIpAddr != 0) lblGIpAddr->setText(new_address);
}

void MainWindow::createTabs() {
    // Create everything!

    // First up, the tab widget
    tabWidget = new QTabWidget(this);

    // Next - the General tab. Its parent is set to this to isolate it
    // from the tabWidget which could get destroyed at any time.
    generalWidget = new QWidget(tabWidget);

    // Next, the General Tabs layout
    vblGeneralLayout = new QVBoxLayout(generalWidget);
    generalWidget->setLayout(vblGeneralLayout);

    // And then the first Group box (Connection)
    gbConnection = new QGroupBox(generalWidget);
    gbConnection->setTitle("Connection");
    vblGeneralLayout->addWidget(gbConnection);

    // And its layout
    glConnectionLayout = new QGridLayout(gbConnection);

    // And then the Connection Group Boxes content
    lblGStatusLabel = new QLabel(gbConnection);
    lblGStatusLabel->setText("Status:");
    glConnectionLayout->addWidget(lblGStatusLabel,0,0);

    lblGStatus = new QLabel(gbConnection);
    lblGStatus->setText("Disconnected");
    lblGStatus->setAlignment(Qt::AlignRight);
    glConnectionLayout->addWidget(lblGStatus,0,1);

    lblGIPAddrLabel = new QLabel(gbConnection);
    lblGIPAddrLabel->setText("IP-Address:");
    glConnectionLayout->addWidget(lblGIPAddrLabel,1,0);

    lblGIpAddr = new QLabel(gbConnection);
    lblGIpAddr->setText("0.0.0.0");
    lblGIpAddr->setAlignment(Qt::AlignRight);
    glConnectionLayout->addWidget(lblGIpAddr,1,1);

    lblGIFaceLabel = new QLabel(gbConnection);
    lblGIFaceLabel->setText("Interface:");
    glConnectionLayout->addWidget(lblGIFaceLabel,2,0);

    lblGIFace = new QLabel(gbConnection);
    lblGIFace->setText("none");
    lblGIFace->setAlignment(Qt::AlignRight);
    glConnectionLayout->addWidget(lblGIFace,2,1);


    // Then the second Group Box (Activity)
    gbActivity = new QGroupBox(generalWidget);
    gbActivity->setTitle("Activity");
    vblGeneralLayout->addWidget(gbActivity);

    // And its layout
    glActivityLayout = new QGridLayout(gbActivity);

    // And then its content
    lblGOutLabel = new QLabel(gbActivity);
    lblGOutLabel->setText("Current Sent:");
    glActivityLayout->addWidget(lblGOutLabel,0,0);

    lblGOut = new QLabel(gbActivity);
    lblGOut->setText("0 B/s (~0 B/s)");
    lblGOut->setAlignment(Qt::AlignRight);
    glActivityLayout->addWidget(lblGOut,0,1);

    lblGInLabel = new QLabel(gbActivity);
    lblGInLabel->setText("Current Received:");
    glActivityLayout->addWidget(lblGInLabel,1,0);

    lblGIn = new QLabel(gbActivity);
    lblGIn->setText("0 B/s (~0 B/s)");
    lblGIn->setAlignment(Qt::AlignRight);
    glActivityLayout->addWidget(lblGIn,1,1);

    // And a spacer
    vblGeneralLayout->addStretch(100);

    // And lastly, add the General tab to the tab widget
    tabWidget->addTab(generalWidget,"General");

    tabs_created = true;
}

void MainWindow::tabEnabledStateChanged() {
    qDebug() << "Tab Enabled state changed - Reconfiguring GUI...";
    // If the tabs don't already exist, create them now
    if (!tabs_created &&
        SettingsManager::isTabWidgetEnabled()) {
        qDebug() << "Creating new widgets...";
        createTabs();
    }

    if (SettingsManager::isTabWidgetEnabled()) {

        tabWidget->addTab(centralWidget, "Graph");

        // Set the new central widget
        setCentralWidget(tabWidget);

    } else {

        // And then we make the isolation widget the central one
        setCentralWidget(centralWidget);
    }
    qDebug() << "Done.";
}
