/*
-----------------------------------------------------
   Application Name: QNetTraf
          File Name: host.cpp
         Start Date: 6/12/2009
       Last Changed: $LastChangedDate: 2010-01-19 18:12:52 +1300 (Tue, 19 Jan 2010) $
           Revision: $Revision: 179 $
          Author(s): David Goodwin
          Copyright: (C) Copyright David Goodwin, 2009, 2010
            License: GNU General Public License
   File Description: nettrafd connection implementation
-----------------------------------------------------
        This file is part of QNetTraf.

    This is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This software is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this software; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

File Notes:
------------------
*/
#include "host.h"

Host::Host(QString hostname,
           int port,
           QString password,
           QString iface,
           int timeout,
           bool newif)
{
    // Store parameters
    m_iface = iface;
    m_hostname = hostname;
    m_port = port;
    m_iface = iface;
    m_timeout = timeout;
    m_passwd = password;
    m_newif = newif;

    // Reset everthing.
    ibr_total = 0;
    obr_total = 0;
    m_samples = 0;
    ibr_average = 0;
    obr_average = 0;
    max_ibr = 0;
    max_obr = 0;
    ibr = 0;
    obr = 0;

    socket_ptr = new QTcpSocket();
}

Host::~Host() {
    delete socket_ptr;
}

bool Host::hostConnect() {
    // Reset everything

    QString response;

    socket_ptr->connectToHost(m_hostname, m_port, QIODevice::ReadWrite);

    if (socket_ptr->waitForConnected(m_timeout)) {
        // We have managed to connect!

        // Now we need to send the password.
        socket_ptr->write(QByteArray("PASS " + m_passwd.toAscii()));
        socket_ptr->waitForReadyRead(m_timeout);

        // Get the response
        response = QString(socket_ptr->readLine());
        qDebug() << response;

        if (!response.startsWith("PASS OK")) {
            // The password was not accepted.
            qDebug("Invalid Password");
            QMessageBox::warning(0,"Error: PADPASS","The supplied password was not accepted.");
            return false;
        }

        // If we got this far, the password was accepted. Connect up everything.
        connect(socket_ptr, SIGNAL(readyRead()), this, SLOT(dataArrived()));
        connect(socket_ptr,SIGNAL(disconnected()),this,SLOT(socketDisconnected()));

        // Set the interface
        if (m_newif)
            socket_ptr->write(QByteArray("NITF " + m_iface.toAscii()));
        else
            socket_ptr->write(QByteArray("ITF " + m_iface.toAscii()));

        return true;
    }
    else {
        qDebug("Connect Failed.");
        return false;
    }
}

bool Host::hostDisconnect() {
    socket_ptr->disconnectFromHost();
    return true;
}

void Host::socketDisconnected() {
    emit disconnected();
}

void Host::dataArrived() {
    QString data;
    data = QString(socket_ptr->readLine());

    qDebug() << "Data Recvd: " << data;
    QStringList fields;

    if (data.startsWith("IPADDR ")) {
        // We are being told our IP address.
        fields = data.split(" ");
        Q_ASSERT(fields.count() >= 2);
        ip_address = fields.at(1);
        emit ipset(ip_address.trimmed());
    }
    else if (data.startsWith("NDATA ")) {
        // We are being told the data rate.
        fields = data.split(" ");
        Q_ASSERT(fields.count() >= 3);

        // Get the incoming and outoing data rate
        ibr = fields.at(1).toLong();
        obr = fields.at(2).toLong();

        // Update the running total
        ibr_total += ibr;
        obr_total += obr;
        m_samples++;

        // Update the average
        ibr_average = ibr_total / m_samples;
        obr_average = obr_total / m_samples;

        // Update the maximum seen so far
        if (max_ibr < ibr)
            max_ibr = ibr;
        if (max_obr < obr)
            max_obr = obr;

        qDebug() << "NEmit: " << QString::number(ibr) << QString::number(obr);
        emit activity(ibr, obr);
    }
    else if(data.startsWith("DATA ")) {
        // We are being told the data received so far.
        fields = data.split(" ");
        Q_ASSERT(fields.count() >= 3);

        long iD = fields.at(1).toLong();
        long oD = fields.at(2).toLong();

        qDebug() << "RETR: " << QString::number(iD) << QString::number(oD);
        qDebug() << "IBTO: " << QString::number(ibr_total) << QString::number(obr_total);

        if (ibr_total == 0)
            ibr = iD;
        else
            ibr = iD - ibr_total;

        if (obr_total == 0)
            obr = oD;
        else
            obr = oD - obr_total;

        qDebug() << "CALC: " << QString::number(ibr) << QString::number(obr);

        ibr_total += ibr;
        obr_total += obr;
        m_samples++;

        ibr_average = ibr_total / m_samples;
        obr_average = obr_total / m_samples;

        if (max_ibr < ibr)
            max_ibr = ibr;
        if (max_obr < obr)
            max_obr = obr;

        qDebug() << "Emit: " << QString::number(ibr) << QString::number(obr);
        emit activity(ibr,obr);
    }
    else if (data.startsWith("ERROR")) {
        // An error has occured
        emit error();
    }
    else {
        // We have probably had an error.
        qDebug() << "Unknown command received: " << data;
    }
}

void Host::quit() {
    socket_ptr->write(QByteArray("QUIT"));
}

void Host::die() {
    socket_ptr->write(QByteArray("DIE"));
}
