/*
-----------------------------------------------------
   Application Name: QNetTraf
          File Name: settingsmanager.cpp
         Start Date: 16/01/2010
       Last Changed: $LastChangedDate: 2010-01-19 18:12:52 +1300 (Tue, 19 Jan 2010) $
           Revision: $Revision: 179 $
          Author(s): David Goodwin
          Copyright: (C) Copyright David Goodwin, 2009, 2010
            License: GNU General Public License
   File Description: Settings Manager Implementation
-----------------------------------------------------
        This file is part of QNetTraf.

    This is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This software is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this software; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

File Notes:
------------------
*/

#include "settingsmanager.h"

QSettings *SettingsManager::settings;

SettingsManager::SettingsManager(QObject *parent) :
    QObject(parent)
{
    settings = 0;
}

void SettingsManager::loadSettings() {
    if (settings != 0) closeSettings();

    // See if appname.ini exists. If so, use that instead.
    QString filename;
    filename = QCoreApplication::applicationName() + ".ini";

    if (QFile::exists(filename)) {
        // Load settings from there
        settings = new QSettings(filename, QSettings::IniFormat, 0);
    } else {
        settings = new QSettings();
    }
}

void SettingsManager::closeSettings() {
    if (settings != 0) {
        delete settings;
        settings = 0;
    }
}

QString SettingsManager::getHostName() {
    if (settings == 0) return "";
    return settings->value("/connection/hostname","").toString();
}

void SettingsManager::setHostName(QString hostname) {
    if (settings == 0) return;
    settings->setValue("/connection/hostname",hostname);
}

int SettingsManager::getPort() {
    if (settings == 0) return 4664;
    return settings->value("/connection/port",4664).toInt();
}

void SettingsManager::setPort(int port) {
    if (settings == 0) return;
    settings->setValue("/connection/port",port);
}

QString SettingsManager::getInterfaceName() {
    if (settings == 0) return "";
    return settings->value("/connection/interface_name","eth0").toString();
}

void SettingsManager::setInterfaceName(QString iface) {
    if (settings == 0) return;
    settings->setValue("/connection/interface_name",iface);
}

int SettingsManager::getProtocolVersion() {
    if (settings == 0) return 1;
    return settings->value("/connection/protocol_version",1).toInt();
}

void SettingsManager::setProtocolVersion(int version) {
    if (settings == 0) return;
    settings->setValue("/connection/protocol_version",version);
}

QString SettingsManager::getPassword() {
    if (settings == 0) return "";
    return settings->value("/connection/password","").toString();
}

void SettingsManager::setPassword(QString password) {
    if (settings == 0) return;
    settings->setValue("/connection/password",password);
}

bool SettingsManager::isLogDataEnabled() {
    if (settings == 0) return false;
    return settings->value("/pref/log_data",false).toBool();
}

void SettingsManager::setLogDataEnabled(bool enabled) {
    if (settings == 0) return;
    settings->setValue("/pref/log_data",enabled);
}

bool SettingsManager::isTitlebarSpeedEnabled() {
    if (settings == 0) return false;
    return settings->value("/pref/titlebar_speed",false).toBool();
}

void SettingsManager::setTitlebarSpeedEnabled(bool enabled) {
    if (settings == 0) return;
    settings->setValue("/pref/titlebar_speed",enabled);
}

bool SettingsManager::isSystemTrayHintShown() {
    if (settings == 0) return false;
    return settings->value("/pref/hints/systray",false).toBool();
}

void SettingsManager::setSystemTrayHintShown(bool enabled) {
    if (settings == 0) return;
    settings->setValue("/pref/hints/systray",enabled);
}

int SettingsManager::getSysTrayMessageTimeout() {
    if (settings == 0) return 15000;
    return settings->value("/pref/ui/message_timeout",15000).toInt();
}

void SettingsManager::setSysTrayMessageTimeout(int timeout) {
    if (settings == 0) return;
    settings->setValue("/pref/ui/message_timeout",timeout);
}

int SettingsManager::getConnectionRetryTimeout() {
    if (settings == 0) return 5000;
    return settings->value("/connection/reconnect_timeout",5000).toInt();
}

void SettingsManager::setConnectionRetryTimeout(int timeout) {
    if (settings == 0) return;
    settings->setValue("/connection/reconnect_timeout",timeout);
}

bool SettingsManager::isHelpEnabled() {
    if (settings == 0) return false;
    return settings->value("/pref/ui/help_enabled",false).toBool();
}

void SettingsManager::setHelpEnabled(bool enabled) {
    if (settings == 0) return;
    settings->setValue("/pref/ui/help_enabled",enabled);
}

bool SettingsManager::isTabWidgetEnabled() {
    if (settings == 0) return true;
    return settings->value("/pref/ui/tab_widget_enabled",true).toBool();
}

void SettingsManager::setTabWidgetEnabled(bool enabled) {
    if (settings == 0) return;
    settings->setValue("/pref/ui/tab_widget_enabled",enabled);
}

bool SettingsManager::isMinimiseOnCloseEnabled() {
    if (settings == 0) return true;
    return settings->value("/pref/ui/minimise_on_close",true).toBool();
}

void SettingsManager::setMinimiseOnCloseEnabled(bool enabled) {
    if (settings == 0) return;
    settings->setValue("/pref/ui/minimise_on_close",enabled);
}

bool SettingsManager::showConfigDialog() {
    if (settings == 0) return true;
    return settings->value("/show_config_dialog",true).toBool();
}

void SettingsManager::setShowConfigDialog(bool enabled) {
    if (settings == 0) return;
    settings->setValue("/show_config_dialog",enabled);
}

QString SettingsManager::getStatusIconStyle() {
    if (settings == 0) return "redblue";
    return settings->value("/pref/ui/activity_icon_style","redblue").toString();
}

void SettingsManager::setStatusIconStyle(QString style) {
    if (settings == 0) return;
    settings->setValue("/pref/ui/activity_icon_style",style);
}

QString SettingsManager::getMessageValue(QString message_name, QString value_name, QString def_message) {
    if (settings == 0) return def_message;

    QString key = "/messages/" + message_name + "/" + value_name;
    return settings->value(key,def_message).toString();
}

QString SettingsManager::getReconnectedMessageTitle() {
    QString def_msg = "Connection Reestablished";
    QString message_name = "reconnected";
    QString value_name = "title";
    return getMessageValue(message_name,value_name,def_msg);
}

QString SettingsManager::getReconnectedMessageBody() {
    QString def_msg = "RNetTraf has reestablished its "
                      "connection to the host being monitored.";
    QString message_name = "reconnected";
    QString value_name = "body";
    return getMessageValue(message_name,value_name,def_msg);
}

QString SettingsManager::getReconnectedMessageURL() {
    QString def_msg = "";
    QString message_name = "reconnected";
    QString value_name = "url";
    return getMessageValue(message_name,value_name,def_msg);
}

QString SettingsManager::getFailedConnectMessageTitle() {
    QString def_msg = "Unable to Connect";
    QString message_name = "connect_failed";
    QString value_name = "title";
    return getMessageValue(message_name,value_name,def_msg);
}
QString SettingsManager::getFailedConnectMessageBody() {
    QString def_msg =  "RNetTraf is unable to connect to the "
                       "specified host. It is possible that the host is temporarily "
                       "unresponsive or unavailable. The connection attempt will be "
                       "automatically retried.";
    QString message_name = "connect_failed";
    QString value_name = "body";
    return getMessageValue(message_name,value_name,def_msg);
}
QString SettingsManager::getFailedConnectMessageURL() {
    QString def_msg = "";
    QString message_name = "connect_failed";
    QString value_name = "url";
    return getMessageValue(message_name,value_name,def_msg);
}

QString SettingsManager::getDisconnectedMessageTitle() {
    QString def_msg = "Connection Lost";
    QString message_name = "disconnected";
    QString value_name = "title";
    return getMessageValue(message_name,value_name,def_msg);
}
QString SettingsManager::getDisconnectedMessageBody() {
    QString def_msg = "RNetTraf has lost its connection to "
                      "the host being monitored. RNetTraf will try to reconnect.";
    QString message_name = "disconnected";
    QString value_name = "body";
    return getMessageValue(message_name,value_name,def_msg);
}

QString SettingsManager::getDisconnectedMessageURL() {
    QString def_msg = "";
    QString message_name = "disconnected";
    QString value_name = "url";
    return getMessageValue(message_name,value_name,def_msg);
}

QString SettingsManager::getInvalidInterfaceMessageBody(bool always_invalid) {
   QString def_msg;
   QString message_name;
   if (always_invalid) {
       def_msg = "The network interface RNetTraf is monitoring is invalid. It is "
                 "possible that the network interface does not exist "
                 "or the network interface is down. RNetTraf will "
                 "continue monitoring and notify if it becomes valid.";
       message_name = "invalid_interface";
   } else {
       def_msg = "The network interface RNetTraf was monitoring has just become "
                 "invalid. It is likely that the network interface has gone down. "
                 "RNetTraf will continue to monitor the interface and notify when "
                 "it becomes valid again.";
       message_name = "now_invalid_interface";
   }

   QString value_name = "body";
   return getMessageValue(message_name,value_name,def_msg);
}

QString SettingsManager::getInvalidInterfaceMessageURL(bool always_invalid){
    QString def_msg;
    QString message_name;
    if (always_invalid) {
        def_msg = "";
        message_name = "invalid_interface";
    } else {
        def_msg = "";
        message_name = "now_invalid_interface";
    }
    QString value_name = "body";
    return getMessageValue(message_name,value_name,def_msg);
 }

QString SettingsManager::getNowValidMessageTitle() {
    QString def_msg = "Interface Available";
    QString message_name = "now_valid";
    QString value_name = "title";
    return getMessageValue(message_name,value_name,def_msg);
}
QString SettingsManager::getNowValidMessageBody() {
    QString def_msg = "The previously invalid interface "
                      "has just become valid again. Network traffic through "
                      "this interface should now be able to resume.";
    QString message_name = "now_valid";
    QString value_name = "body";
    return getMessageValue(message_name,value_name,def_msg);
}
QString SettingsManager::getNowValidMessageURL() {
    QString def_msg = "";
    QString message_name = "now_valid";
    QString value_name = "url";
    return getMessageValue(message_name,value_name,def_msg);
}
