/*
-----------------------------------------------------
   Application Name: QNetTraf
          File Name: graphscene.cpp
         Start Date: 6/12/2009
       Last Changed: $LastChangedDate: 2010-01-19 18:12:52 +1300 (Tue, 19 Jan 2010) $
           Revision: $Revision: 179 $
          Author(s): David Goodwin
          Copyright: (C) Copyright David Goodwin, 2009, 2010
            License: GNU General Public License
   File Description: Graph Widget implementation
-----------------------------------------------------
        This file is part of QNetTraf.

    This is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This software is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this software; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

File Notes:
------------------
*/

#include "graphscene.h"

GraphScene::GraphScene(QWidget *parent): QWidget(parent)
{
    supplied_maximum_value = 0;
    running_total = 0;
    local_max_value = 0;
    min_value = LONG_MAX;

    // Default pen colour
    m_pen = QPen(Qt::blue);

    // Default background colour
    m_background = Qt::black;

    // Head line
    m_head_line_enabled = true;
    m_head_line_pen = QPen(Qt::gray);

    // Use MiB or MB?
    m_use_i = true;

    // Average Display
    m_average_display_enabled = true;

    // Speed display
    m_speed_display_enabled = true;
    m_min_speed_display_enabled = true;
    m_max_speed_display_enabled = true;

    // Gradient control
    m_enable_gradient = true;

    // The format string used for printing out floating-point numbers
    m_std_dbl_format_string = "%.2f";
    m_min_dbl_format_string = "%.3g";

    QSizePolicy qsp;
    qsp.setHorizontalPolicy(QSizePolicy::MinimumExpanding);
    qsp.setVerticalPolicy(QSizePolicy::MinimumExpanding);
    setSizePolicy(qsp);
}

void GraphScene::addSample(long value) {
    qDebug() << "GraphScene::add_sample(value: " + QString::number(value) + ")";

    // First up: a bit of sanity checking
    if (value < 0) {
        qWarning() << "Sample value less than zero.";
        value = 0;
    }

    /* Here we need to:
       1. Append the sample to the list of samples and trim the graph
       2. Compute the scale to draw the lines at
       3. Regenerate all lines applying the computed scale
       4. Redraw all lines
    */


    // 1. Append sample to the list of samples
    samples.append(value);

    // Trim the graph
    if (samples.count() > width()) {
        // We need to remove samples from the start of the graph until it is
        // no longer too small
        while (samples.count() > width()) {
            running_total -= samples.takeFirst();
        }
    }
    qDebug() << "******************************* SAMPLES: " << samples.count();

    // 2. Compute the scale to draw the lines at
    float scale = calculate_scale();

    // 3. Regenerate all lines applying the computed scale
    rescale(scale);

    // Compute the average
    running_total += value;
    average = (running_total / samples.count());
    qDebug() << "Running Total: " << running_total << " Sample Count: " << samples.count();
    qDebug() << "Average: " << average << " Scale factor: " << scale << " Scaled Average:" << average * scale;
    scaled_average = average * scale;

    // 4. Redraw all lines
    repaint();
}

float GraphScene::calculate_scale() {
    qDebug() << "GraphScene::calculate_scale()";

    /**** Ok. This function calculates the appropriate scale factor for this
      graph so that all bars remain fully visible at all times. Nothing gets
      drawn off the top. To calculate this, we need to know the maximum
      value that we have to fit into the graph.

      However, we may want to sync this graphs scale factor with another. For
      that purpose we emit our maximum value when ever it changes and the
      ability is provided for our client to override our own maximum value.

      Doing so allows the maximum values of two or more graphs to be synced
      allowing all graphs to use the same maximum value for scale computations.

      So, we end up with two maximum values:
        supplied_maximum_value   -> A value that has been given to us
        local_max_value          -> The max value in _this_ graph
      For scale factor computations we always use the greater of the two.
     */

    /* This function calculates the scale to draw graph bars (lines) at. The
       scale should be such that the tallest bar in the graph reaches to the
       top of the graph widget and no higher.

       1. Find the maximum value in the graphs range
       2. Get the height of the graph widget
       3. Compute and return the scale
     */


    // 1. Calculate the minimum and maximum values currently in the graph
    long max_value = 0;
    for (int i = 0; i < samples.count(); i++) {
        long current_value = samples.at(i);
        // Update the max value
        if (current_value > max_value)
            max_value = current_value;
        // Update the min value
        if (current_value < min_value)
            min_value = current_value;
    }
if (max_value == 0) return 1.0;
    old_max_value = local_max_value;
    local_max_value = max_value;

    // If the max value has changed, announce it to the world
    emit maximum_value_changed(max_value);

    // Now we make a decision on what maximum value to use - our local one
    // or one that has been provided to us. We will use the greater of the
    // two.
    if (max_value < supplied_maximum_value)
        max_value = supplied_maximum_value;

    // Cant divide by 0...
    if (max_value == 0) return 1.0;

    // 2. Get the height of the graph widget
    const int widget_height = this->height();

    // 3. Compute the scale
    qDebug() << "Widget Height: " << widget_height;
    qDebug() << "Max Value: " << max_value;
    double scale = (double)widget_height / (double)max_value;
    qDebug() << "Scale: " << scale;

    return scale;
}

void GraphScene::rescale(float scale_factor) {
    qDebug() << "GraphScene::rescale() scale_factor: " << QString::number(scale_factor);
    /* This function regenerates all lines in the graph from the list of
       samples and scales them according to the supplied scale_factor.

       1. Clear the list of lines
       2. Generate the lines
     */

    // 1. Clear the list of lines
    lines.clear();

    // 2. Generate the list of lines
    qDebug() << "Rescaling...";
    const int sample_count = samples.count();
    const int widget_height = height();
    for (int i = 0; i < sample_count; i++) {
        const long sample = samples.at(i);

        const int scaled_value = (int)((double)sample * scale_factor);
        //qDebug() << "Sample: " << sample << "Scaled Value: " << scaled_value;
        const int bar_top = (widget_height - scaled_value);

        QLine line(i,widget_height,i,bar_top);
        lines.append(line);
    }
}

void GraphScene::paintEvent(QPaintEvent *) {
    qDebug() << "GraphScene::paintEvenet()";

    QPainter painter(this);

    // Set the background colour:
    painter.setBrush(m_background);
    painter.drawRect(0,0,width(),height());

    // Draw the graph itself
    for (int i = 0; i < lines.count(); i++) {
        const QLine current_line = lines.at(i);
        painter.setPen(get_graph_colour(i));
        painter.drawLine(current_line);
    }

    // Draw the leading edge line
    if (m_head_line_enabled && (lines.count() < width())) {
        qDebug() << "Head-Line enabled";
        int xloc = lines.count();
        painter.setPen(m_head_line_pen);
        painter.drawLine(xloc,0,xloc,height());
    }

    QRectF box(0,0,width(),height());
    QRectF lower_rec(0,height()-13,width(),8);
    QRectF top_rec(0,0,width(),8);

    QFont std_font = painter.font();
    QFont small_font = painter.font();
    QFont original_font = painter.font();
    small_font.setPointSize(7);
    std_font.setPointSize(10);

    // Draw the Average
    if (m_average_display_enabled) {
        int average_pos = height() - scaled_average;

        if (m_help_enabled) {
            //QRectF avgLineText(average_pos - 20,0,width(),20);
            QRectF avgLineText(0,average_pos-10,width(),10);
            painter.setFont(small_font);
            painter.setPen(QPen(Qt::white));
            painter.drawText(avgLineText,Qt::AlignLeft, "Average");
        }

        painter.setPen(QPen(Qt::black));
        painter.drawLine(0,average_pos,width(),average_pos);
        painter.setPen(m_head_line_pen);
        QList<QLine> qvl = make_dashed_line(0,width(),average_pos,3);
        for (int i = 0; i < qvl.count(); i++)
            painter.drawLine(qvl.at(i));

        print_speed(&painter,average,m_head_line_pen,lower_rec,
                    small_font,Qt::AlignVCenter | Qt::AlignCenter, "Average");
    }

    if (!samples.isEmpty()) {
        // Current speed
        if (m_speed_display_enabled)
            print_speed(&painter,samples.last(),QPen(Qt::white),box,
                    std_font,Qt::AlignVCenter | Qt::AlignCenter, "Current");
        // Max Value
        if (m_max_speed_display_enabled)
            print_speed(&painter,local_max_value,QPen(Qt::darkYellow),top_rec,
                    small_font, Qt::AlignVCenter | Qt::AlignRight, "Max");
        // Min Value
        if (m_min_speed_display_enabled)
            print_speed(&painter,min_value,QPen(Qt::darkYellow),lower_rec,
                    small_font,Qt::AlignVCenter | Qt::AlignRight, "Min");
    }
}

void GraphScene::print_speed(QPainter *painter, double speed, QPen pen,
                             QRectF rect, QFont font, int flags, QString name) {

    QFont old_font = painter->font();
    QPen old_pen = painter->pen();
    painter->setFont(font);
    painter->setPen(pen);

    // Convert the average value into the appropriate units:
    QString *units = new QString();
    double *converted_value = new double;
    *converted_value = speed;
    convert_size(converted_value,units, m_use_i);

    QString format_string;
    if (units->compare("B") == 0)
        format_string = m_min_dbl_format_string;
    else
        format_string = m_std_dbl_format_string;

    QString text;
    text.sprintf(format_string.toAscii().constData(),*converted_value);
    text = text + " " + *units + "/s";

    if (m_help_enabled) text = name + ": " + text;

    painter->drawText(rect,flags, text);
    painter->setFont(old_font);
    painter->setPen(old_pen);

    delete units;
    delete converted_value;
}

QList<QLine> GraphScene::make_dashed_line(int start_x, int end_x, int y, int dash_length) {
   QList<QLine> qvl;

   for (int i = start_x; i < end_x; i += dash_length*2) {
       QLine line(i,y,i+dash_length-1,y);
       qvl.append(line);
   }
   return qvl;
}

void GraphScene::convert_size(double * size, QString * units, bool i) {
    double avg_speed = *size;
    QString unit = "B";
    if (avg_speed > 1024) {
        avg_speed /= 1024;
        if (i) unit = "KiB"; else unit = "KB";
    }
    if (avg_speed > 1024) {
        avg_speed /= 1024;
        if (i) unit = "MiB"; else unit = "MB";
    }
    if (avg_speed > 1024) {
        avg_speed /= 1024;
        if (i) unit = "GiB"; else unit = "GB";
    }
    if (avg_speed > 1024) {
        avg_speed /= 1024;
        if (i) unit = "TiB"; else unit = "TB";
    }
    *size = avg_speed;
    *units = unit;
}

QPen GraphScene::get_graph_colour(int sample) {
    // First up, if gradients are off - just return the pen.
    if (!m_enable_gradient) return m_pen;

    // Ok. So gradients are on. Lets see if we can actually do a gradient. We
    // only support single colour gradients so if two or more of RGB are set
    // then just return the pen as is.
    int i = 0;
    int c = 0;
    int v = 0;
    QColor color = m_pen.color();
    if (color.red() > 0) {
        i++;
        c = 1;
        v = color.red();
    }
    if (color.green() > 0) {
        i++;
        c = 2;
        v = color.green();
    }
    if (color.blue() > 0) {
        i++;
        c = 3;
        v = color.blue();
    }
    if (i != 1) return m_pen;

    // Ok. So only one colour is set (either red, green or blue).
    // Lets compute the gradient!
    int nv = v + (155 / samples.count()+1) * sample;

    if (nv > 255) nv = 255;

    switch (c) {
        case 1:
            color.setRed(nv);
            break;
        case 2:
            color.setGreen(nv);
            break;
        case 3:
            color.setBlue(nv);
            break;
        default:
            return m_pen;
    }
    QPen new_pen(color);
    return new_pen;
}

void GraphScene::reset() {
    samples.clear();
    lines.clear();

    supplied_maximum_value = 0;
    running_total = 0;
    local_max_value = 0;
    min_value = LONG_MAX;
    repaint();
}

QSize GraphScene::sizeHint() const {
    return QSize(100,100);
}
