/*
 * This Software (C) Copyright David Goodwin, 2008, 2009.
 *
 *   This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef FSPLUGIN_H
#define FSPLUGIN_H

#include <QIODevice>
#include <QString>
#include <QByteArray>
#include <QList>
#include <QFlags>

/* We need this to resolve a circular dependency:
      FSPlugin needs to know about FSEntry because two of its methods
        return them.
      FSEntry needs to know about FSPlugin because some of its methods
        deal with an enumeration defined in FSPlugin.
 */
class FSEntry;

/** FSPlugin provides an interface to Filesystem Plugins for DIMG and its
 * related programs. This includes QFATreader and DIMG itself.
 */
class FSPlugin {

public:

    /** Features supported by this plugin.
     */
    enum Feature {
        F_NoFeatures   = 0x00000000, /*!< No features are supported (shouldnt be used) */
        F_DisplayAbout = 0x00000001, /*!< Indicates that the displayAbout() method has been implemented */
        F_Configure    = 0x00000002, /*!< Indicates that the configure() method has been implemented */
        F_ReadOnly     = 0x00000004, /*!< Indicates that the Filesystem is Read-Only */
        F_VolumeLabel  = 0x00000008, /*!< Indicates that the volumeLabel() method will return a Volume Label */
        F_CreationDate = 0x00000010, /*!< File Creation Date is recorded */
        F_CreationTime = 0x00000020, /*!< File Creation Time is recorded */
        F_AccessDate   = 0x00000040, /*!< Last access date of file is recorded */
        F_AccessTime   = 0x00000080, /*!< Last access time of file is recorded */
        F_ModDate      = 0x00000100, /*!< File Modification Date is recorded */
        F_ModTime      = 0x00000200  /*!< File Modification Time is recorded */
    };
    Q_DECLARE_FLAGS(Features, Feature)

    enum Attribute {
        /* Attributes Supported by FAT12: */
        A_NoAttributes = 0x0000,  /*!< No attributes are set */
        A_ReadOnly     = 0x0001,  /*!< Read-Only File Attribute */
        A_Hidden       = 0x0002,  /*!< Hidden File Attribute */
        A_System       = 0x0004,  /*!< System File Attribute */
        A_VolumeID     = 0x0008,  /*!< VolumeID Attribute (Entry is the filesystem Volume ID) */
        A_Directory    = 0x0010,  /*!< Directory Attribute (Entry is a Directory) */
        A_Archive      = 0x0020   /*!< Archive Attribute */
    };
    Q_DECLARE_FLAGS(Attributes,Attribute)

    virtual ~FSPlugin() {} /*!< Does Nothing. */

    /** This function is called to load the filesystem from the
        specified device. When this function returns, the filesystem
        should be ready to use.

        If a filesystem is already open, this should return False.

        This virtual function must be implemented by all subclasses.

        \param dev This is a pointer to the QIODevice that the filesystem will
                    work off. Only Random Access devices are supported.
        \returns True if the filesystem was able to load successfully. If there
                 was an error, False should be returned.
    */
    virtual bool load(QIODevice * dev) = 0;

    /** This function is called when the filesystem should be closed. This must
        be called before the filesystem can be loaded from another QIODevice.

        This virtual function must be implemented by all subclasses.

        \returns True if filesystem could be closed. If there were any errors,
                 this function should return False.
     */
    virtual bool close() = 0;

    /** This function returns true if the current state is closed,
      otherwise it returns false.
      \returns True if the filesystem is currently in the closed state.
      */
    virtual bool isClosed() = 0;

    /** This should display or output some form of about screen for this
        plugin.

        The default implementation does nothing - if a subclass does not
        reimplement it, this should indicated by the flags returned by
        the getFeatures() method.

        This function will only ever be called if getFeatures() indicates that
        it has been implemented.
     */
    virtual void displayAbout() const { }

    /** This should display some form of configuration screen if one is
        required by this plugin. The default implementation does nothing.

        This function will only ever be called if getFeatures() indicates that
        it has been implemented.
     */
    virtual void configure() { }

    /** This method should return a version string for this plugin. Something
        like "1.9.0-beta3" would be acceptable. It is only used for display
        purposes.

        This virtual function must be implemented by all subclasses.

        \returns A string containing the version of this plugin.
     */
    virtual QString getVersionString() const = 0;

    /** This method should return the name of the plugin. Something like
        "FAT12 Filesystem" would be acceptable. It is only used for display
        purposes.

        When combined with getVersionString(), the output would look something
        like:
           FAT12 Filesystem version 1.9.0-beta3

        \returns A string containing the name of this plugin.
     */
    virtual QString getPluginName() const = 0;

    /** Returns the string passed to setPluginFilename(QString). Subclasses
        should not need to reimplement this.

        \returns The string passed to setPluginFilename(QString).
     */
    virtual QString plugin_filename() const {return m_plugin_filename;}

    /** Sets the plugins filename. plugin_filename() should return what ever
        string is passed into this function.

        \param fn This plugins filename (eg, fat12.dll)
     */
    virtual void setPluginFilename(QString fn) {m_plugin_filename = fn;}

    /** This function is called when the host program is exiting. The default
        implementation just calls close().

        This virtual function should probably be implemented by all subclasses
        if any form of tidyup is required.

        Once this function has been called, there is no guarantee that any
        other functions will be called. The program will exit sometime shortly
        after this has been called.
     */
    virtual void exiting() { close(); }

    /** This function indicates which features the Filesystem supports.
     * The information it returns is used to decide which features the
     * plugin supports. For example, if the F_DisplayAbout flag is not set
     * then the DisplayAbout method will never be called.
     */
    virtual Features getFeatures() const = 0;

    /** This function indicates which file attributes the Filesystem supports.
      No data structure ever returned by a subclass should make use of
      attributes not specified in the value this function returns.

      If a data structure returned by a subclass uses an attribute that is not
      returned by this method, it is a bug.

      \returns All attributes returned by this filesystem (Attributes that are
               only used internally dont need to be returned here).
     */
    virtual Attributes getSupportedAttributes() const = 0;

    /** Returns the filesystems Volume Label if supported by the filesystem.
        This method will only be called if the F_VolumeLabel flag is returned
        by the getSupportedAttributes() method.

        \returns The filesystems Volume Label.
     */
    virtual QString volumeLabel() const = 0;

    /** Returns the error-code for the last error that occured.
      \returns The error-code for the last error that occured.
     */
    virtual int getLastErrorNumber() const = 0;

    /** Returns the error string for the last error that occured.
      \returns The error string for the last error that occured.
     */
    virtual QString getLastErrorString() const = 0;

    /** Copies a file from this filesystem into the computers filesystem.

        For example, if a subclass implemented this interface over a zip file,
        this method would be the extract operation. When called, it would
        extract the specified file from the zip file to the specified location
        on a mounted disk on the local system.

        \param source_filename The name of the file in this filesystem to copy.
                    It takes any filename that is valid for this filesystem
        \param destination_filename The name of the destination file that the
                 source file should be copied to. It takes any filename that
                 is valid for the local system.
        \returns True if the copy operation was successfull, False if there was
                 an error.
     */
    virtual bool copyFile(QString source_filename, QString destination_filename) = 0;

    /** Reads a file into memory and returns it.

        Because this method loads the entire file into memory, it should only
        be used for small files. A Read method may be provided at some point in
        the future when more complex filesystems arise using this interface.

        \param filename
     */
    virtual QByteArray readFile(QString filename) = 0;

    /** Returns an FSEntry that represents the specified file or directory.

      The FSEntry returned by this function should represent the specified
      filename. If the specified file could not be found, it should just return
      null.

      It is the responsibility of the caller to delete the FSEntry returned
      by this method. The FSPlugin will not keep track of it.

      \param filename The name of the file to get the FSEntry for.
      \returns The FSEntry that represents the specified file or directory.
     */
    virtual FSEntry * getEntryByName(QString filename) = 0;

    /** Returns a list of all FSEntries in the specified directory. If the
      directory could not be found, it should return an empty list.

      It is the responsibility of the caller to delete every FSEntry returned
      by this method. The FSPlugin will not keep track of them.

      \param directory The name of the directory to get an entry listing for.
      \returns An entry listing for the specified directory.
     */
    virtual QList<FSEntry *> getEntryList(QString directory) = 0;

protected:
    /** Stores the string passed into setPluginFilename(QString) and the string
        returned by plugin_filename().
     */
    QString m_plugin_filename;

};

Q_DECLARE_OPERATORS_FOR_FLAGS(FSPlugin::Features)
Q_DECLARE_OPERATORS_FOR_FLAGS(FSPlugin::Attributes)

Q_DECLARE_INTERFACE(FSPlugin, "com.dgt.DIMG.FSPlugin/1.0");

#endif // FSPLUGIN_H
