/* Project "DIMG" Drive Access Driver Plugin Interface
 *
 * This Software (C) Copyright David Goodwin, 2008, 2009.
 *
 *   This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef _DQDRIVER_H
#define _DQDRIVER_H

// Qt Headers
#include <QString>
#include <QStringList>
#include <QByteArray>
#include <QFlags>

// Plugin Headers
#include <DiskGeometry.h>

/** This is version 1 of the drive access driver interface for QtDIMG. It
  allows floppy drive access routines to be moved out of the main program
  to keep coupling to a minimum and make long-term maintenance easier.

  Some drivers may require geometry information in order to translate logical
  sectors into physical sectors. Those that do should have the feature flag
  F_GEOM_CHS or F_GEOM_BOOTSEC set. These flags indicate how disk geometry
  information should be supplied to the driver. If F_GEOM_CHS is set, geometry
  information should be supplied in the form of the DiskGeometry structure to
  the setDiskGeom(DiskGeometry) method before writing data to the disk. When
  reading from a disk, setting the disk geometry should not be required.

  If F_GEOM_BOOTSEC is set, The boot sector should be passed into the driver
  for it to decode and figure out the geometry from.

  If F_GEOM_NR is set this driver does not require any disk geometry
  information - most likely because it can either figure it out on its own or
  such information does not make sense.
 */
class DQDskDriver {

public:
    /** Features supported by plugins
     */
    enum Feature {
        F_NoFeatures   = 0x00000000, /*!< No features are supported (shouldn't be used) */
        F_DisplayAbout = 0x00000001, /*!< Indicates that the displayAbout() method has been implemented */
        F_Configure    = 0x00000002, /*!< Indicates that the configure() method has been implemented */
        F_GEOM_NR      = 0x00000004, /*!< Disk Geometry information is not required/not supported/doesnt make sense. */
        F_GEOM_CHS     = 0x00000008, /*!< Driver supports CHS geometry information (Reserved for future use) */
        F_GEOM_BOOTSEC = 0x00000010, /*!< Driver needs to obtain disk geometry information from the boot sector */
        F_READ_ONLY    = 0x00000020, /*!< Drives can only be read from */
        F_WRITE_ONLY   = 0x00000040  /*!< Drives can only be written to */
    };
    Q_DECLARE_FLAGS(Features, Feature)

    virtual ~DQDskDriver() {}   /*!< Does Nothing. */

    /** Gets the features supported by this plugin
     */
    virtual Features getFeatures() const = 0;

    /** Initialises the plugin. This is called shortly after loading and should return success (0) or an error number.
      \returns 0 on success or any other number on error
     */
    virtual int initialise() = 0;

    /** Displays an about screen. This is only ever called if the F_DisplayAbout feature is reported.
     */
    virtual void displayAbout() const {}

    /** Displays a configuration dialog. This is only ever called if the F_Configure feature is reported.
     */
    virtual void configure() {}

    /** This should return a human-readable version string (eg, "1.0.0").
      \returns A version string
     */
    virtual QString getVersionString() const = 0;

    /** This should return a human-readable name for the plugin (eg, "LibDsk Drive Access Plugin")
      \returns The plugin name
     */
    virtual QString getPluginName() const = 0;

    /** This is used by the host program to get the last error number that occured.
      \returns The error code for the last error that occured.
     */
    virtual int getLastErrorNumber() const = 0;

    /** This is used by the host program to get a description of the error which can be shown to the user.
      \param errorCode The error code to get a description for
      \returns A string describing the error
     */
    virtual QString errToString(int errorCode) const = 0;

    /** Gets a list of available drives that this plugin can access. The drive names
        returned should be accepted by the openDrive(QString) method.
        \returns A list of available drive names.
     */
    virtual QStringList getDriveNames() const = 0;

    /** Gets a list of descriptive names for drives available on the system to be displayed
      in the host programs User Interface. The ordering of this list should be the same as
      that of getDriveNames() (as in, entry 0 in both lists should refer to the same device).
     */
    virtual QStringList getDescriptiveDriveNames() const = 0;

    /** Opens the specified drive and makes it ready for read or write access. If successfull the
      drive should be fully ready for reading or writing once this call has completed.
      \param driveName The name of the drive to open. This should be one of the strings returned
         by getDriveNames().
      \returns 0 for success or any other number for failure.
     */
    virtual int openDrive(QString driveName) = 0;

    /** Closes the current drive making it no longer available for reading or writing.
      \returns 0 for success or any other number for failure.
     */
    virtual int closeDrive() = 0;

    /** Returns the name of the currently open drive (if any)
      \returns The drive name string for the currently device (if any) or "" if none is open.
     */
    virtual QString currentDrive() const = 0;

    /** Returns True if there is a drive open, False otherwise.
      \returns True if a drive is open or False otherwise.
     */
    virtual bool isOpen() const = 0;

    /** This method method returns if the specified device is writable or not.
      This may only check if the drive itself is capable of writing (for example,
      a CD-ROM drive can not write). It does not have to check if the media in the
      drive is writable.
      \param driveName the drive to check the writability of.
      \returns True if the drive is writable or False if it is read only.
     */
    virtual bool driveIsWritable(QString driveName) const = 0;

    /** Attempts to write the specified data to the currently open disk at the
      specified position (sector). This function will never be called if
      driveIsWritable(QString) has returned False for the currently open drive.

      If the drive is writable but the media is not (eg, a write-protected
      floppy disk), this method should just return an error.

     \param sector The sector to write the data to.
     \param data The data to write at the specified sector.
     \returns 0 for success or any other number for an error (including read-only media).

     */
    virtual int writeData(int sector, const QByteArray *data) = 0;

    /** Attempts to read the specified logical sector into the supplied QByteArray.
      Should there be any problems (read errors, no media, etc) it should
      return an error but the device should be left in such a state as to
      allow the operation to be retried.

      \param sector The sector to read
      \param data The QByteArray to read the sector into.
     */
    virtual int readData(int sector, QByteArray *data) = 0;

    /** Attempts to read the specified sector into the supplied QByteArray.
      Should there be any problems (read errors, no media, etc) it should
      return an error but the device should be left in such a state as to
      allow the operation to be retried.

      This iwll only ever be called if physical (CHS) sectoring is reported as
      being supported by this plugin.

      \param cylinder The cylinder number
      \param head The head number
      \param sector The sector number
      \param data The QByteArray to read the sector into.
     */
    //virtual int readData(int cylinder, int head, int sector, QByteArray *data) = 0;

    /** Called when the host program is exiting. The plugin should use this to
      do any tidying up required including making sure drives are closed, etc.

      Once this method has returned no other methods in the plugin will be called
      before it is unloaded.
     */
    virtual void exiting() { }

    /** Gets the disk geometry for the media loaded in the currently open drive.
      If there is a problem obtaining the disk geometry information it should just
      return an empty DiskGeometry instance.

      \returns An instance of DiskGeomtry initialised with the geometry information
      for the media in the currently open drive or an empty DiskGeometry instance
      if the information could not be obtained.
     */
    virtual DiskGeometry getDiskGeom() = 0;

    /** This method sets the disk geometry to be used when writing to the currently open drive. It
      is only ever called if the F_GEOM_BOOTSEC feature is set and the F_GEOM_NR feature is _not_ set.

      The method should try to figure out the correct disk geometry settings based on the first sector
      of a valid image for the media type in the currently open drive. Should the method be unable to
      determine the disk geometry based on the information in the boot sector it should return an error.

      \param boot_sector The first sector of an image from which disk geometry should be obtained.
      \returns 0 for success or any other number for failure.
     */
    virtual int setDiskGeom(QByteArray *boot_sector) {return 0; }

    /** This method sets the disk geometry to be used when writing to the currently open drive. It
      is only ever called if the F_GEOM_CHS feature is set and the F_GEOM_NR and F_GEOM_BOOTSEC
      features are _not_ set.

      It should take an instance of the DiskGeometry structure and use it when writing to the
      device.

      \param dg The Disk Geometry structure to use
      \returns 0 for success or any other number for failure (including if the DiskGeometry
      structures information is insufficient).
     */
    virtual int setDiskGeom(DiskGeometry dg) { return 0; }  // Reserved for future use
};

Q_DECLARE_OPERATORS_FOR_FLAGS(DQDskDriver::Features)
Q_DECLARE_INTERFACE(DQDskDriver, "com.dgt.DIMG.DQDskDriver/1.0");

#endif
