#include "fat12fs.h"
#include "fat12entry.h"

bool Fat12fs::load(QIODevice * dev){
    // If there isnt a filesystem, make one
    if (fs == 0) fs = new FAT12();

    // Load the filesystem
    fs->load(dev);
    is_closed = false;
    return true;
}

bool Fat12fs::close(){
    if (fs == 0) return false;
    fs->close();
    is_closed = true;
    return true;
}

FSPlugin::Features Fat12fs::getFeatures() const{
    // At the moment we dont support Configure.

    FSPlugin::Features f;

    // First, the filesystem features:
    f |= FSPlugin::F_ReadOnly | FSPlugin::F_VolumeLabel;

    // Then the GUI features:
    f |= FSPlugin::F_DisplayAbout;

    // Then the file date/time features:
    f |= FSPlugin::F_CreationDate | FSPlugin::F_CreationTime;
    f |= FSPlugin::F_AccessDate;  // Access time isnt stored
    f |= FSPlugin::F_ModDate | FSPlugin::F_ModTime;

    // Then return it.
    return f;
}

FSPlugin::Attributes Fat12fs::getSupportedAttributes() const{
    /* FAT12 supports the ReadOnly, Hidden, System, VolumeID, Directory and
       Archive attributes.

       However, the VolumeID attribute should only be used internally by the
       filesystem - DOS never exposes it.
     */
    FSPlugin::Attributes a;

    a = a | A_ReadOnly | A_Hidden | A_System | A_Directory | A_Archive;

    return a;
}

QString Fat12fs::volumeLabel() const {
    return fs->volumeLabel();
}

int Fat12fs::getLastErrorNumber() const{
    return 0;
}

QString Fat12fs::getLastErrorString() const{
    return "";
}

bool Fat12fs::copyFile(QString source_filename, QString destination_filename){
    return fs->extractFile(source_filename, destination_filename);
}

QByteArray Fat12fs::readFile(QString filename){
    // TODO: Implement this
    return QByteArray();
}

FSEntry * Fat12fs::getEntryByName(QString filename){
    QStringList qsl = filename.split("/",QString::SkipEmptyParts);
    QString file = qsl.takeLast();
    QString dir = qsl.join("/");

    // Ok, we are looking for [file] in [dir]
    QList<FAT12_DirEntry> entrylist = fs->dir(dir);

    // If the directory couldnt be found, return null
    if (entrylist.isEmpty()) return 0;

    // Ok, we have a directory listing. Now we need to find
    // the specified entry.
    for (int i = 0; i < entrylist.size(); i++) {
        FAT12_DirEntry e = entrylist.at(i);
        if (e.fullFileName() == file) {
            // We have found the desired entry!
            return new Fat12Entry(e, dir);
        }
    }

    // Seems we couldnt find the entry. Return null.
    return 0;
}

QList<FSEntry *> Fat12fs::getEntryList(QString directory){
    // Get the directory listing.
    QList<FAT12_DirEntry> el = fs->dir(directory);

    // If the directory listing was empty, return an empty entry list.
    if (el.isEmpty()) return QList<FSEntry *>();

    // Now we need to convert all the entires
    QList<FSEntry *> entryList;
    while (!el.isEmpty()) {
        FAT12_DirEntry e = el.takeFirst();
        if (e.filename() != "." && e.filename() != "..")
            entryList.append(new Fat12Entry(e,directory));
    }
    return entryList;
}

void Fat12fs::displayAbout() const {
    about_dialog ad;
    ad.exec();
}

Q_EXPORT_PLUGIN2(fat12fs, Fat12fs)
