#ifndef FAT12_FAT_H
#define FAT12_FAT_H

#include <QObject>
#include <QIODevice>
#include <QByteArray>
#include <QList>
#include <QMessageBox>
#include <QProgressDialog>
#include <QFile>
#include "fat12_bpb.h"
#include "fat12_direntry.h"
#include "utils.h"

/** The File Allocation Table. This handles loading file data from the
    filesystem. It is essentially the entire filesystem minus functions for
    processing directory data.
 */
class FAT12_FAT : public QObject
{

public:
    /** Constructs a new File Allocation Table. It loads the first
        File Allocation Table from the supplied QIODevice using information
        supplied in the BIOS Parameter Block.
        \param file The device to load the FAT from
        \param BIOS_parameter_block The BIOS Parameter Block which will provide the
                         details required to find and load the FAT.
     */
    FAT12_FAT(QIODevice * file, FAT12_BPB *BIOS_parameter_block);
    ~FAT12_FAT(); /*!< Destructor */

    /** Loads an entire file from the filesystem into memory and returns it as
        a series of bytes.

        This method takes the start cluster for the desired file and then loads
        all of the files data into memory. Because of this behaviour, this
        function is only useful for small files.

        There is not currently any supplied way of dealing with any large files.

        \param startCluster The address for the desired files first cluster.
        \returns The full files data.
     */
    QByteArray loadFile(int startCluster);

    /** Extracts the specified file in this filesystem to a local file.

      \param e The directory entry to extract
      \param destfile The destination file to extract the data to
      \returns True for success, False for Failure.
     */
    bool extractFile(FAT12_DirEntry e, QString destfile);

    /** Loads the chain of clusters for the file starting with the specified
        start cluster.

        \param startCluster The first cluster in the file
        \returns A list of all the clusters in the file
     */
    QList<int> getChain(int startCluster);

    /** Gets the value for the specified cluster.

      \param clusterNumber the cluster number to get the value for.
      \returns The value for the specified cluster.
     */
    int getValue(int clusterNumber);

private:
    QByteArray *FAT_data;   /*!< The FAT's data */
    QIODevice  *iod;        /*!< The QIODevice for loading file data */
    FAT12_BPB  *bpb;        /*!< The BIOS PArameter Block */
    int FAT_location;       /*!< The FAT's location */
    int FAT_size;           /*!< The size of the FAT */
};

#endif // FAT12_FAT_H
