#include "fat12_fat.h"

FAT12_FAT::FAT12_FAT(QIODevice * file, FAT12_BPB * BIOS_parameter_block) {
    // First, store the inputs
    iod = file;
    bpb = BIOS_parameter_block;

    /* Now we must figure out where the FAT we want is. Its location depends
       on which one we want. The first one should begin straight after the
       reserved secotrs at the start of the disk. The second FAT (if there
       is one) should be at:
            reserved_sectors * bytes_per_sector
            + fat_number * sectors_per_fat * bytes_per_sector
     */

    // To use a different FAT, change this number to the one we want.
    int fat_number = 0;
    int reserved_bytes = bpb->reserved_sectors * bpb->bytes_per_sector;
    int fat_bytes = fat_number * (bpb->sectors_per_fat * bpb->bytes_per_sector);

    // Store its location and size
    FAT_location = reserved_bytes + fat_bytes;
    FAT_size = (bpb->sectors_per_fat * bpb->bytes_per_sector);

    // Now load it into memory for fast and easy access.

    if (!iod->seek(FAT_location)) {
        //QMessageBox::critical(this,"Seek Error",
        //        "Unable to seek to the File Allocation Table. Filesystem or image may be corrupt.", QMessageBox::Ok, QMessageBox::Ok);
    }
    FAT_data = new QByteArray(iod->read(FAT_size));

    // And we should be done now.
}

FAT12_FAT::~FAT12_FAT() {
    delete FAT_data;
    // iod is passed in and so should not be deleted here - the object that passed it
    //      in probably still has it.
    // bpb is passed in and so should not be deleted here - the object that passed it
    //      in probably still has it.
}

QByteArray FAT12_FAT::loadFile(int startCluster) {
    /* First we must get the chain. This chain tells us which clusters
     * we need to load up
     */
    QList<int> chain = getChain(startCluster);

    /* Ok. Now we need to compute where the data section starts. This is
     * equal to:
     *      Number of Reserved Sectors
     *    + Number of FATs * Size of a FAT
     * which is then multiplied by the number of bytes in a sector.
     */
    int data_offset = bpb->reserved_sectors;
    data_offset += bpb->total_fats * bpb->sectors_per_fat;
    data_offset *= bpb->bytes_per_sector;

    /* Now we must add the space allocated to the directory table to the
     * offset
     */
    data_offset += bpb->max_root_entries * 32;

    /* Next we need to allocate some memory for the file data. The address
     * of each cluster is stored in the <chain> variable so the number
     * of bytes we need is the number of clusters times the number of
     * sectors per cluster.
     */
    int file_data_size = chain.size();
    file_data_size *= bpb->sectors_per_cluster;
    file_data_size *= bpb->bytes_per_sector;

    QByteArray file_data;
    for (int i = 0; i < file_data_size; i++)
        file_data.append('\0');


    /* We now need to loop over all the clusters in the chain. For each
     * cluster we need to compute its address, read however many
     * sectors are in a cluster into the file data array
     */
    for (int i = 0; i < chain.size(); i++) {
        // Calculate the start point for the data in the data block
        int current_cluster = chain.at(i);
        int data_start = data_offset;
        data_start += ((current_cluster - 2) * bpb->sectors_per_cluster) * bpb->bytes_per_sector;

        // Calculate how much data this cluster contains
        int cluster_data_size = bpb->sectors_per_cluster * bpb->bytes_per_sector;

        // We need somewhere to store this cluster
        QByteArray cluster_data;

        // Figure out where in our file_data array this cluster should go
        int cluster_location = i * bpb->sectors_per_cluster * bpb->bytes_per_sector;

        // Read the data in from the file
        if (!iod->seek(data_start)) {
            //QMessageBox::critical(this,"Seek Error",
            //          "Unable to seek to the requested file or directory. Filesystem or image may be corrupt.", QMessageBox::Ok, QMessageBox::Ok);
        }
        cluster_data = iod->read(cluster_data_size);

        // Now shift the cluster_data into the array
        file_data.replace(cluster_location, cluster_data_size, cluster_data);
    }

    return file_data;
}

bool FAT12_FAT::extractFile(FAT12_DirEntry e, QString destfile) {

    // Get the entry cluster
    int startCluster = e.getEntryCluster();

    int bytesToWrite = e.getSize();

    QProgressDialog progress("Extracting " + e.fullFileName(),"Abort Extraction", 0, bytesToWrite,0);


    QFile f(destfile);

    if (!f.open(QIODevice::WriteOnly)) {
        QMessageBox::warning(0,"Unable to open destination file",
                             "Unable to open destination file for writing");
        return false;
    }

    /* First we must get the chain. This chain tells us which clusters
     * we need to load up
     */

    QList<int> chain = getChain(startCluster);

    /* Ok. Now we need to compute where the data section starts. This is
     * equal to:
     *      Number of Reserved Sectors
     *    + Number of FATs * Size of a FAT
     * which is then multiplied by the number of bytes in a sector.
     */
    int data_offset = bpb->reserved_sectors;
    data_offset += bpb->total_fats * bpb->sectors_per_fat;
    data_offset *= bpb->bytes_per_sector;

    /* Now we must add the space allocated to the directory table to the
     * offset
     */
    data_offset += bpb->max_root_entries * 32;

    /* Next we need to allocate some memory for the file data. The address
     * of each cluster is stored in the <chain> variable so the number
     * of bytes we need is the number of clusters times the number of
     * sectors per cluster.
     */
    //int file_data_size = chain.size();
    //file_data_size *= bpb->sectors_per_cluster;
    //file_data_size *= bpb->bytes_per_sector;

    /* We now need to loop over all the clusters in the chain. For each
     * cluster we need to compute its address, read however many
     * sectors are in a cluster into the file data array
     */

    for (int i = 0; i < chain.size(); i++) {
        // Calculate the start point for the data in the data block
        int current_cluster = chain.at(i);
        int data_start = data_offset;
        data_start += ((current_cluster - 2) * bpb->sectors_per_cluster) * bpb->bytes_per_sector;

        // Calculate how much data this cluster contains
        int cluster_data_size = bpb->sectors_per_cluster * bpb->bytes_per_sector;

        // We need somewhere to store this cluster
        QByteArray cluster_data;

        // Figure out where in our file_data array this cluster should go
        int cluster_location = i * bpb->sectors_per_cluster * bpb->bytes_per_sector;

        // Read the data in from the file
        if (!iod->seek(data_start)) {
            //QMessageBox::critical(this,"Seek Error",
            //          "Unable to seek to the requested file or directory. Filesystem or image may be corrupt.", QMessageBox::Ok, QMessageBox::Ok);
        }
        cluster_data = iod->read(cluster_data_size);

        if (!f.seek(cluster_location)) {
            QMessageBox::critical(0,"Seek Error", "Seek failed on output file.");
            f.close();
            return false;
        }
        
        int written;
        if (bytesToWrite < cluster_data.size())
            written = f.write(cluster_data,bytesToWrite);
        else
            written = f.write(cluster_data);

        bytesToWrite -= written;

        progress.setValue(progress.value() + written);

        if (progress.wasCanceled()) {
            f.close();
            return true;
        }
        
        /*if (written != cluster_data_size) {
            QMessageBox::warning(0,"Write Error", "The expected number of bytes was not written to the output file. The file may not be completely extracted.");
            f.close();
            return false;
        }*/
    }

    f.close();
    progress.setValue(e.getSize());

    return true;
}

QList<int> FAT12_FAT::getChain(int startCluster) {
    QList<int> chain;

    /* Here we need to get a chain of clusters. Each value we get points
     * to the next value. The final value is 0xFFF which is where we stop.
     * This value isnt added to the list.
     */
    int current_value = startCluster;
    while (current_value != 0xFFF) {
        // add the value
        chain.append(current_value);

        // Then get the next one
        current_value = getValue(current_value);
    }

    return chain;
}

int FAT12_FAT::getValue(int clusterNumber) {

    /* I tried writing this function three times and none of them worked! It
       seems that the details I could find on implementing this function were
       either:
         a) wrong
         b) Easy to interpret incorrectly

       Seems I was able to implement all read-only functionality of FAT12
       except this one function.

       In the end, I just grabbed and adapted the code below from libfat
       version 0.3. The specific file was libfat.c and the original function
       was:
         static int fat12_read_entry(Volume_t *V, DWORD N, int FatNum, DWORD *Value)
       the only changes I have made were replacing stuff like
       DWORD, WORD and BYTE with their actual types. And, of course, inserted
       this programs data structures in the place of the ones libfat uses.
     */

    unsigned short val;
    if (clusterNumber%2 == 0) {
        val = 0;

        ((unsigned char *) &val)[0] = (unsigned char) FAT_data->at(((int) (clusterNumber * 1.5)));
        ((unsigned char *) &val)[1] = (unsigned char) FAT_data->at(((int) (clusterNumber * 1.5)) + 1);

        ((unsigned char *) &val)[1] &= 0x0F;
    } else {
        val =0;

        ((unsigned char *) &val)[0] = (unsigned char) FAT_data->at(((int) (clusterNumber * 1.5)));
        ((unsigned char *) &val)[1] = (unsigned char) FAT_data->at(((int) (clusterNumber * 1.5)) + 1);

        ((unsigned char *) &val)[0] = (((unsigned char *) &val)[0] >> 4);
        ((unsigned char *) &val)[0] |= (((unsigned char *) &val)[1] << 4);
        ((unsigned char *) &val)[1] = (((unsigned char *) &val)[1] >> 4);
    }

    return val;
}
