#ifndef FAT12_DIRENTRY_H
#define FAT12_DIRENTRY_H

#include <QObject>
#include <QString>
#include <QByteArray>
#include <QStringList>
#include <QTime>
#include <QDate>
#include "utils.h"

/** This represents a directory entry in the FAT12 Filesystem. It can represent
 */
class FAT12_DirEntry //: public QObject
{
public:
    FAT12_DirEntry();   /*!< Constructor */
    FAT12_DirEntry(QByteArray data);    /*!< Constructs this FAT12_DirEntry from the supplied data */

    /**
     Loads this FAT12_DirEntry from the supplied data.
     \param data The data to load this directory entry from.
     */
    void load_entry(QByteArray data);


    // Get attributes / properties / values
    /** Returns the read_only attribute
      \returns The Read-Only attribute for this entry
      */
    bool isReadOnly() const       {   return read_only;               }
    /** Returns the hidden attribute
      \returns The Hidden attribute for this entry
      */
    bool isHidden() const         {   return hidden;                  }
    /** Returns the System-File attribute
      \returns The System-File attribute for this entry
      */
    bool isSystemFile() const     {   return system_file;             }
    /** Returns the Volume ID attribute
      \returns The Volume ID attribute for this entry
      */
    bool isVolumeID() const       {   return volume_id;               }
    /** Returns the Directory attribute
      \returns The Directory attribute for this entry
      */
    bool isDirectory() const      {   return directory;               }
    /** Returns the Archive attribute
      \returns The Archive attribute for this entry
      */
    bool isArchive() const        {   return archive;                 }

    /** Returns the filename for this entry. That is the name and the extension
        combined without the `.'

        If the name is "Readme" and the extension is "txt", this will return
        "Readmetxt". It is useful for obtaining directory names. To obtain
        normal filenames, fullFileName() should be used.

        \returns The filename followed by the extension.
     */
    QString filename() const      {   return m_filename.trimmed() + ext.trimmed();        }

    /** Returns the full filename for this entry. That is the name, followed by
      the `.' character and then the extension.

      If the name is "Readme" and the extension is "txt", this will
      return "Readme.txt"
      \returns the filename, followed by `.' followed by the extension.
     */
    QString fullFileName() const;

    /** Returns the entry cluster for this file.
      \returns The Entry Cluster for this file.
     */
    int getEntryCluster() const   {   return entryCluster;            }

    /** Returns the size of this file. If this file is a directory, the size
      will be reported as 0.
      \returns The Size of this file.
     */
    int getSize() const           {   return fileSize;                }

    /** Returns a duplicate of this entry.
      \returns A Duplicate of this entry.
     */
    FAT12_DirEntry copy();

    /** Returns a string-representation of this directory entry as it would
      appear in a directory listing.
      \returns A string-representation of this entry as it would appear in a
                directory listing.
     */
    QString toString();
    /** Returns a list of strings for this entry as they would be used in a
        directory listing. Their order is:
            -# File/Directory name
            -# Attributes (of the form rhsavd where non-set attributes are output as a -)
            -# Size in the appropriate unit
            -# Entry Cluster number
     */
    QStringList toValues();

    /** Returns the time when the file was created.

      \returns The time when the file was created.
      */
    QTime creationTime() const { return create_time; }
    /** Returns the time when the file was last modified.
      \returns The time when the file was last modified.
      */
    QTime modificationTime() const {return modified_time; }

    /** Returns the date when the file was created.
       \returns The date when the file was created.
       */
    QDate creationDate() const { return create_date; }
    /** Returns the date when the file was last accessed.
      \returns The date when the file was last accessed.
      */
    QDate accessedDate() const { return access_date; }
    /** Returns the date when the file was last modified.
      \returns The date when the file was last modified.
      */
    QDate modificationDate() const { return modified_date; }

private:
    // Methods
    void loadAttributes(char attributes);   /*<! Loads the attributes for this entry */

    /** Reads the MS-DOS Timestamp out of the two supplied bytes and converts
      it to a QTime.

      \param byteA The first byte of the MS-DOS Timestamp
      \param byteB The second byte of the MS-DOS Timestamp
      \returns The MS-DOS Timestamp converted to a QTime.
     */
    QTime convertTime(char byteA, char byteB);

    /** Reads the MS-DOS Datestamp out of the two supplied bytes and converts
      it to a QDate

      \param byteA The first byte of the date
      \param byteB The second byte of the date
      \returns The MS-DOS Date converted to a QDate.
     */
    QDate convertDate(char byteA, char byteB);

    QByteArray entry_data;      /*<! The data that this entry was loaded from */

    // String values
    QString m_filename;         /*!< The filename */
    QString ext;                /*!< The extension */
    QString date;               /*!< The date */
    QString time;               /*!< The time */
    char attributes;            /*!< The byte defining the attributes */
    int entryCluster;           /*!< The entry cluster number */
    int fileSize;               /*!< The file size in bytes */


    // Attributes
    bool read_only;             /*!< The Read-Only attribute */
    bool hidden;                /*!< The Hidden attribute */
    bool system_file;           /*!< The System-File attribute */
    bool volume_id;             /*!< The Volmue ID attribute */
    bool directory;             /*!< The Directory attribute */
    bool archive;               /*!< The Archive attribute */

    // Dates and Times
    QTime create_time;
    QDate create_date;
    QDate access_date;
    QTime modified_time;
    QDate modified_date;
};

#endif // FAT12_DIRENTRY_H
