#ifndef FAT12_DIRECTORY_H
#define FAT12_DIRECTORY_H

#include <QObject>
#include <QList>
#include <QIODevice>
#include <QString>
#include <QByteArray>
#include <QtDebug>
#include "fat12_direntry.h"
#include "fat12_bpb.h"
#include "fat12_fat.h"

/** Represents a directory in the FAT12 Filesystem.
 */
class FAT12_Directory : public QObject
{
public:
    FAT12_Directory();      /*!< Constructor */
    ~FAT12_Directory();     /*!< Destructor */

    /** Starts the recursive Tree Load. This functions purpose is to
        read out the data for the root directory and then pass it to
        loadTree(QByteArray,QIODevice,FAT12_BPB,FAT12_FAT,QString). What ever
        FAT12_Directory this method is called on will be the root directory for
        a new tree.

        \param iod The QIODevice to load the Root Directory from
        \param start The start of the Root Directory
        \param table_length Reserved (not currently used)
        \param bpb The BIOS Parameter Block for this Filesystem
        \param fat The File Allocation Table for this filesystem
     */
    void loadTree(QIODevice *iod,
                  long start,
                  int table_length,
                  FAT12_BPB *bpb,
                  FAT12_FAT *fat);

    /** Loads this directory from the supplied data.
        It performs a recursive call to itself passing in the data for each
        subdirectory. When this method is called, this directory and all
        subdirectories will be loaded into a subtree.

        This method should normally only be called by itself and
        loadTree(QIODevice,long,int,FAT12_BPB,FAT12_FAT).

        \param data The data to load this tree from. It is a sequence of 32bit
                    directory entry fields.
        \param iod The QIODevice to load subdirectories from.
        \param bpb The BIOS Parameter Block for this filesystem.
        \param fat The File Allocation Table for this filesystem.
        \param name The name of this directory.
     */
    void loadTree(QByteArray data,
                  QIODevice *iod,
                  FAT12_BPB *bpb,
                  FAT12_FAT *fat,
                  QString name);

    /** Returns a list of all entries in this directory. If this directory is
        empty it will return an empty list.

        \returns A list of all files and subdirectories inside this directory.
     */
    QList<FAT12_DirEntry> getEntries();

    /** Returns the subdirectory specified.

      \param name The name of the subdirectory to return.
      \returns The requested subdirectory or NULL if it could not be found in
                this directory.
     */
    FAT12_Directory * getSubdirByName(QString name);

    /** Returns the name of this directory.
      \returns The name of this directory.
     */
    QString getName() { return directory_name; }

private:
    QList<FAT12_DirEntry *> * entries;  /*!< A list of all entries in this directory. */
    QList<FAT12_Directory *> * subdirs; /*!< A list of all subdirectories. */
    FAT12_BPB *BIOS_parameter_block;    /*!< The BIOS Parameter Block for this filesystem. */
    FAT12_FAT *fat;                     /*!< The File Allocation Table for this filesystem. */
    QIODevice *iod;                     /*!< The QIODevice for this filesystem. */
    QString directory_name;             /*!< The name of this directory. */
};

#endif // FAT12_DIRECTORY_H
