#ifndef FAT12_H
#define FAT12_H

#include <QObject>
#include <QtDebug>
#include <QIODevice>
#include <QString>
#include <QList>
#include <QMessageBox>
#include "fat12_bpb.h"
#include "fat12_fat.h"
#include "fat12_directory.h"
#include "fat12_direntry.h"

/**
 A Read-Only implementation of Microsoft Corporations FAT12 Filesystem as
 commonly used on Floppy Disks.
 */
class FAT12 : public QObject
{
public:
    FAT12();    /*!< Constructor. */
    ~FAT12();   /*!< Destructor. */

public slots:
    /** Loads the filesystem from the specified QIODevice.
     * This causes the File Allocation Table, BIOS Parameter Block and the
     * the entire directory tree to be loaded into memory for fast and easy
     * access.
     *
     * \param device The QIODevice to load the filesystem from. This must
     *               support random access.
     */
    void load(QIODevice * device);

    /** Closes the filesystem. This must be called before load(QIODevice) can
        be called again.
     */
    void close();

    /** Returns a directory listing for the specified directory.

      \param directory The directory to get a listing for. For example, "/dir1/dir2/".
      \returns A QList containing one FAT12_DirEntry for each file in the directory.
     */
    QList<FAT12_DirEntry> dir(QString directory);

    /** Outputs some debugging information using qDebug().
     */
    void printDebugInfo();

    /** Ouputs a basic directory listing of the specified directory using qDebug().

      \param directory The directory to list.
     */
    void printDebugDirListing(QString directory);

    /** Returns the volume label. If no file with the volume label attribute
      set is found in the root directory, the volume label from the BIOS
      Parameter Block will be used.

      \returns The Volume Label.
      */
    QString volumeLabel() const;


    /** Extracts the specified internal file to the specified external file.

      \param source The source file inside this filesystem
      \param dest The destination file on this computer
      \returns True for success, False for Failure.
     */
    bool extractFile(QString source, QString dest);

private:
    // private methods

    /** Gets the FAT12_Directory representing the specified directory name.
      \param directory The directory to get.
      \returns The FAT12_Directory for the specified directory.
     */
    FAT12_Directory * get_directory(QString directory);

    // private variables
    QIODevice *iod;     /*!< The currently used QIODevice */
    FAT12_BPB *bpb;     /*!< The BIOS Parameter Block */
    FAT12_FAT *first_fat;   /*!< The first File Allocation Table */
    bool bootWord;      /*!< If the Boot Word was present */
    FAT12_Directory * root_directory;   /*!< The Root Directory */
    QString VolumeLabel; /*!< Stores the filesystems Volume Label */
};

#endif // FAT12_H
