#include "mainwindow.h"
#include "ui_mainwindow.h"

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent), ui(new Ui::MainWindowClass)
{
    ui->setupUi(this);
    connect(ui->action_Open, SIGNAL(triggered()),this,SLOT(open()));
    connect(ui->actionE_xit, SIGNAL(triggered()),this,SLOT(exit()));
    connect(ui->action_About,SIGNAL(triggered()),this,SLOT(show_app_about()));
    p = 0;
    file = 0;

    // Load the plugin.
    loadPlugin();

    if (p == 0) exit();

    FSPlugin::Features f = p->getFeatures();

    if (!f.testFlag(FSPlugin::F_DisplayAbout)) {
        ui->actionAbout_Plugin->setEnabled(false);
    } else {
        connect(ui->actionAbout_Plugin,SIGNAL(triggered()),this,SLOT(show_about()));
    }

    if (!f.testFlag(FSPlugin::F_Configure)) {
        ui->actionConfigure_Plugin->setEnabled(false);
    } else {
        connect(ui->actionConfigure_Plugin,SIGNAL(triggered()),this,SLOT(show_configure()));
    }

    connect(ui->actionExtract_File, SIGNAL(triggered()),this,SLOT(extract_file()));
    ui->actionExtract_File->setEnabled(false);
    connect(ui->actionFile_Properties,SIGNAL(triggered()),this,SLOT(show_file_properties()));
    ui->actionFile_Properties->setEnabled(false);
}

MainWindow::~MainWindow()
{
    delete ui;
    if (p != 0) delete p;
    if (file != 0) delete file;
}

void MainWindow::loadPlugin() {   
    QString plugin_file, alt_file;

    // Set the directory for the plugin to be the applications directory
    plugin_file = alt_file = QApplication::applicationDirPath();

    // Build the base filename
#ifdef Q_OS_WIN32
    plugin_file += "/fat12fs";
    alt_file += "/fsplugin";
#else
    plugin_file += "/libfat12fs";
    alt_file += "/libfsplugin";
#endif

    // If it is a debug build, use the debug build of the plugin
#ifdef _DIMG_DEBUG
    plugin_file += "d";
    alt_file += "d";
#endif

    // Add on the extension
#ifdef Q_OS_WIN32
    plugin_file += ".dll";
    alt_file += ".dll";
#else
    plugin_file += ".so";
    alt_file += ".so";
#endif

    QFileInfo info(plugin_file);

    if (!info.exists()) {
        plugin_file = alt_file;
        info.setFile(plugin_file);
        if (!info.exists()) {
            plugin_file = QFileDialog::getOpenFileName(this,"Locate Filesystem Plugin");
            if (plugin_file.isEmpty()) {
                QMessageBox::critical(0,"Plugin Load Error","No filesystem plugin specified. Exiting.");
            }
        }
    }

    qDebug() << "Creating plugin loader for file " << plugin_file;
    QPluginLoader *loader = new QPluginLoader(plugin_file);
    qDebug() << "Loading plugin...";
    if (loader->load()) {
        p = qobject_cast<FSPlugin *>(loader->instance());
        if (!p) {
            QMessageBox::critical(0,"Plugin Load Error","Loaded plugin was not of type FSPlugin.");
        }
    }
    else {
        QMessageBox::critical(0,"Plugin Load Error","A filesystem plugin could be found but not loaded.\nThis is likely a problem with the plugin.\nThe Plugin filename is:\n" + plugin_file);
    }
    delete loader;
    loader = 0;

//--------------------------------------
 /*
    QString plugin_file, alt_file;

    // Set the directory for the plugin to be the applications directory
    plugin_file = alt_file = QApplication::applicationDirPath();

#ifdef Q_OS_WIN32
    plugin_file += "/fat12fs.dll";
    alt_file += "/fsplugin.dll";
#else
    plugin_file += "/libfat12fs.so";
    alt_file += "/libfsplugin.so";
#endif

    QFileInfo info(plugin_file);

    if (!info.exists()) {
        plugin_file = alt_file;
        info.setFile(plugin_file);
        if (!info.exists()) {
            plugin_file = QFileDialog::getOpenFileName(this,"Locate Filesystem Plugin");
            if (plugin_file.isEmpty()) {
                QMessageBox::critical(0,"Plugin Load Error","No filesystem plugin specified. Exiting.");
                QApplication::quit();
            }
        }
    }

    //qDebug() << "Creating plugin loader for file " << plugin_file;
    QPluginLoader *loader = new QPluginLoader(plugin_file);
    //qDebug() << "Loading plugin...";
    if (loader->load()) {
        p = qobject_cast<FSPlugin *>(loader->instance());
        if (!p) {
            QMessageBox::critical(0,"Plugin Load Error","Loaded plugin was not of type FSPlugin.");
            QApplication::quit();
        }
    }
    else {
        QMessageBox::critical(0,"Plugin Load Error","Unable to load filesystem plugin");
        QApplication::quit();
    }
    delete loader;*/
}

void MainWindow::open(){
    open(QFileDialog::getOpenFileName(this,"Open Disk Image"));
}

void MainWindow::open(QString filename){
    qDebug() << "Attempting to open file: " << filename;

    qDebug() << "Opening file " << filename;
    file = new QFile(filename);

    if (!file->open(QIODevice::ReadOnly)) {
        // Something has gone wrong
        QMessageBox::critical(0,"status","Unable to open file");
        return;
    }

    if (p == 0) {
        qDebug() << "Loading filesystem plugin...";
        loadPlugin();
        if (p==0) return;
    }

    if (!p->isClosed()) {
        qDebug() << "Closing previous filesystem instance...";
        p->close();
    }

    qDebug() << "Loading filesystem...";
    p->load(file);

    ui->actionExtract_File->setEnabled(true);
    ui->actionFile_Properties->setEnabled(true);

    qDebug() << "Displaying File Tree...";
    ui->treeWidget->clear();
    populateTree(0,"/");

    FSPlugin::Features f = p->getFeatures();

    if (f.testFlag(FSPlugin::F_VolumeLabel)) {
        qDebug() << "Displaying Volume Label...";
        ui->statusBar->showMessage("Volume Label: " + p->volumeLabel());
    }

    this->setWindowTitle("QFATreader - [" + filename + "]");
}

void MainWindow::populateTree(QTreeWidgetItem *parent, QString directory) {
    QList<FSEntry *> entries = p->getEntryList(directory);
    FSPlugin::Features f = p->getFeatures();
    qDebug() << "Entries: " << entries.size();
    for (int i = 0; i < entries.size(); i++) {
        FSEntry *e = entries.at(i);

        // Time to build the strings for the entry:
        // Filename - Attributes (rhasd) - size
        QStringList qsl;
        qsl << e->fileName();

        FSPlugin::Attributes a = e->getAttributes();
        QString attributes;

        // Read-Only:
        if (a.testFlag(FSPlugin::A_ReadOnly))
            attributes.append("r");
        else
            attributes.append("-");

        // Hidden:
        if (a.testFlag(FSPlugin::A_Hidden))
            attributes.append("h");
        else
            attributes.append("-");

        // Archive:
        if (a.testFlag(FSPlugin::A_Archive))
            attributes.append("a");
        else
            attributes.append("-");

        // System File:
        if (a.testFlag(FSPlugin::A_System))
            attributes.append("s");
        else
            attributes.append("-");

        // Directory:
        if (a.testFlag(FSPlugin::A_Directory))
            attributes.append("d");
        else
            attributes.append("-");

        qsl << attributes;

        qsl << QString::number(e->getSize(),10);


        // Check to see if modification date and time are supported.
        if (f.testFlag(FSPlugin::F_ModDate) &&
            f.testFlag(FSPlugin::F_ModTime)) {
            QDate d = e->getDate(FSEntry::D_ModifiedDate);
            QTime t = e->getTime(FSEntry::D_ModifiedTime);
            QString s = d.toString() + " " + t.toString();
            qsl << s;
        }
        // Otherwise, see if at least creation date and time are
        // supported.
        else if (f.testFlag(FSPlugin::F_CreationDate) &&
                 f.testFlag(FSPlugin::F_CreationTime)) {
            QDate d = e->getDate(FSEntry::D_CreatedDate);
            QTime t = e->getTime(FSEntry::D_CreatedTime);
            QString s = d.toString() + t.toString();
            qsl << s;
        }
        // If creation date and time arent supported either, we will just
        // ignore the date and time
        else {
            qsl << "";
        }


        QTreeWidgetItem *qtwi;
        if (parent == 0) {
            qtwi = new QTreeWidgetItem(ui->treeWidget, qsl,0);
        } else {
            qtwi = new QTreeWidgetItem(parent,qsl,0);
        }

        qtwi->setIcon(0,e->getIcon());    
        qtwi->setData(0,Qt::UserRole, e->filePath());

        if (e->isDirectory()) {
            if (e->fileName() != "." && e->fileName() != "..")
                populateTree(qtwi,directory + e->fileName() + "/");
        }
    }

    // Now delete all of the FSEntrys:
    while (!entries.isEmpty())
        delete entries.takeFirst();
}

void MainWindow::exit(){
    if (p != 0) {
        if (!p->isClosed()) p->close();
        p->exiting();
    }

    QApplication::quit();
}

void MainWindow::extract_file(){
    // Get selected filename - Qt::UserRole

    QTreeWidgetItem *qtwi = ui->treeWidget->currentItem();
    if (qtwi == 0) return;

    QString d = qtwi->data(0,Qt::UserRole).toString();
    QString dest_fn = QFileDialog::getSaveFileName(this,"Extract File...",d);
    p->copyFile(d,dest_fn);
}

void MainWindow::show_file_properties() {
    QTreeWidgetItem *qtwi = ui->treeWidget->currentItem();
    if (qtwi == 0) return;

    QString d = qtwi->data(0,Qt::UserRole).toString();
    FSEntry *e = p->getEntryByName(d);
    FileProperties fp;
    fp.setFileEntry(e);
    fp.exec();
    delete e;
}

void MainWindow::show_about() {
    p->displayAbout();
}

void MainWindow::show_configure(){
    p->configure();
}

void MainWindow::show_app_about() {
    QString pn = "No FSPlugin loaded.", pv;

    if (p) {
        pn = p->getPluginName();
        pv = p->getVersionString();
    }

    AboutDialog ad(pn,pv,this);

    ad.exec();

}
