/* Project "DIMG" Windows Direct Driver Plugin
 *
 * This Software (C) Copyright David Goodwin, 2008, 2009.
 *
 *   This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef _DQWindrv_H
#define _DQWindrv_H

// Qt Headers
#include <QString>
#include <QStringList>
#include <QByteArray>
#include <QFlags>
#include <QtPlugin>
#include <QtDebug>
#include <QMessageBox>

// Plugin Headers
#include <DQDriver.h>
#include <DiskGeometry.h>
#include "geom_err.h"

// Windows Headers
#include <windows.h>
#include <winioctl.h>
#include <stdio.h>

// DQWindrv headers
#include "about.h"
#include "constants.h"
#include "geom.h"

/** Windows Direct Drive Access Plugin.
 */
class DQWindrv : public QObject, public DQDskDriver {

    Q_OBJECT
    Q_INTERFACES(DQDskDriver)

public:

    DQWindrv() {}
    ~DQWindrv();

    DQDskDriver::Features getFeatures() const;

    int initialise();

    void displayAbout() const;

    void configure();

    QString getVersionString() const;

    QString getPluginName() const;

    int getLastErrorNumber() const;

    QString errToString(int errorCode) const;

    QStringList getDriveNames() const;

    QStringList getDescriptiveDriveNames() const;

    int openDrive(QString driveName);

    int closeDrive();

    QString currentDrive() const;

    bool isOpen() const;

    bool driveIsWritable(QString driveName) const;

    int writeData(int sector, const QByteArray *data);

    int readData(int sector, QByteArray *data);

    void exiting() { }

    DiskGeometry getDiskGeom();

    int setDiskGeom(QByteArray *boot_sector);
    int setDiskGeom(DiskGeometry dg);

private:
    /** Gets a list of floppy drives attached to the system
      \returns A list of floppy drives (A:, B:, etc) on the system.
     */
    static QStringList getFloppyDrives();

    /** Calls write_out_track() and then processes its success or failure.
      \returns An error code.
     */
    int initiate_track_write();

    /** Writes out the track to disk.
      \returns An error code.
     */
    int write_out_track();

    /** Read in track
      \returns An error code.
     */
    int read_in_track();

    /** Dismounts the drive. After dismounting the OS will automatically try
      to remount it the next time an attempt is made to access it.

      This allows things such as the filesystem to be changed.

      System Requirements: Windows 2000 (Professional or Server) or better

        \param hDisk A handle to a disk
        \returns Success or Failure
     */
    bool DismountVolume(HANDLE hDisk);

    /** Locks the specified volume for exclusive access. A locked volume can be
      accessed only through handles to the file object (*hDevice) that locks
      the volume.

      System Requirements: Windows 2000 (Professional or Server) or better

      \param hDisk A handle to a disk
      \returns Success or Failure
      */
    bool LockVolume(HANDLE hDisk);

    /** Unlocks a volume.

      System Requirements: Windows 2000 (Professional or Server) or better

      \param hDisk A handle to a disk
      \returns Success or Failure
     */
    bool UnlockVolume(HANDLE hDisk);

    /** Retrieve the disks supported media types and store them in
      lpGeometry.

      System Requirements: Windows 2000 (Professional or Server) or better

      \param hDisk A handle to a disk
      \param lpGeometry The Geometry structure to restore the result in.
      \returns Success or Failure
     */
    bool GetDiskGeometry(HANDLE hDisk, PDISK_GEOMETRY lpGeometry);

    /** Gets the floppy drive type.

      \returns 525 for a 5.25" drive, 350 for a 3.5" drive, 2 for Removable Media,
      1 for Fixed Media or 0 otherwise.
     */
    int GetFloppyDriveType(HANDLE hDisk);

    /** Attempts to figure out the media type based on the supplied
      DiskGeometry structure
      \param dg The disk geometry structure
      \returns The media type of the disk
     */
    MEDIA_TYPE GetMediaType(DiskGeometry dg);

    /** Translates a DiskGeometry structure to a windows Disk Geometry
      structure making any required guesses about the Media Type.
      \param dg The Disk Geometry to translate
      \param lpGeometry The windows disk geometry structure to store the
      result in.
      \returns An error code.
     */
    int TranslateDiskGeometry(DiskGeometry dg, PDISK_GEOMETRY lpGeometry);

    /** Gets a string representing the supplied windows error
      \param error The windows error code
      \returns A string representing the supplied windows error code.
     */
    QString GetWindowsErrorString(unsigned long error);

    int lastError;          /*!< The last error to occur */
    DWORD lastWindowsError;  /*!< Last Windows Error */
    QString current_drive;  /*!< Name of the current drive */
    bool drive_open;        /*!< Is the drive open? */

    // For writing
    QByteArray track_buffer; /*!< Track buffer: Stores a track before writing */
    int current_sector;      /*!< The current track sector for writing */
    int fsec;                /*!< If a track write failed, keep track of the
                                  last supplied sector to see if the user
                                  retried */

    // For reading
    QByteArray tread_buffer; /*!< Track buffer: Stores a track for reading */
    int current_read_sector; /*!< The current sector for reading */

    // Other stuff
    DiskGeometry diskgeom;  /*!< Disk Geometry as supplied by DIMG */
    PDISK_GEOMETRY pdg;     /*!< Disk Geometry for Windows */

    // Windows structures
    HANDLE hDrive;      /*!< The current drive handle */
};

#endif
