/* Project "DIMG" Windows Direct Driver Plugin
 *
 * This Software (C) Copyright David Goodwin, 2008, 2009.
 *
 *   This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "DQWindrv.h"

// This stuff is to support autodetecting removable media drives on Windows systems.
#include "windows.h"
static const QString win32_drives[] = {"A", "B", "C", "D", "E", "F",
                                       "G", "H", "I", "J", "K", "L",
                                       "M", "N", "O", "P", "Q", "R",
                                       "S", "T", "U", "V", "W", "X",
                                       "Y", "Z"};
static const int win32_drive_count = 26;

DQDskDriver::Features DQWindrv::getFeatures() const {
    DQDskDriver::Features f;
    f |= DQDskDriver::F_DisplayAbout;
    return f;
}

int DQWindrv::initialise() {
    // Initialise a few variables
    lastError = 0;
    pdg = 0;
    drive_open = false;

    fsec = -1;
    current_read_sector = -2;

    // Check the platform
    OSVERSIONINFOEX osvi;
    ZeroMemory(&osvi,sizeof(osvi));
    osvi.dwOSVersionInfoSize = sizeof(osvi);

    GetVersionEx((OSVERSIONINFO*)&osvi);

    if (osvi.dwPlatformId != VER_PLATFORM_WIN32_NT) {
        QMessageBox::critical(0,"Unsupported Platform","The DQWindrv Windows Direct plugin supports only Windows NT systems.",QMessageBox::Ok);
        return errors::ERR_NOTIMPL;
    }
    return errors::ERR_OK;
}

DQWindrv::~DQWindrv() {
    CloseHandle(hDrive);
}

void DQWindrv::displayAbout() const  {
    About a;
    a.exec();
}

void DQWindrv::configure()  {

}

QString DQWindrv::getVersionString() const {
    return "1.0.0";
}

QString DQWindrv::getPluginName() const {
    return "Windows Direct Driver Plugin";
}

int DQWindrv::getLastErrorNumber() const {
    return lastError;
}

QString DQWindrv::errToString(int errorCode) const {
    switch (errorCode) {
        case errors::ERR_OK:
            return "No Error";
        case errors::ERR_NOTIMPL:
            return "Not Implemented";
        case errors::ERR_USRCANCVIRTMNT:
            return "User Canceled mounting of Virutal Volume";
        case errors::ERR_VOL_LOCK_FAILED:
            return "Failed to lock volume";
        case errors::ERR_VOL_UNLOCK_FAILED:
            return "Failed to unlock volume";
        case errors::ERR_DEV_OPEN_FAILED:
            return "Failed to open device";
        case errors::ERR_DEV_CLOSE_FAILED:
            return "Failed to close device";
        case errors::ERR_NULL_POINTER:
            return "Function was passed a Null Pointer";
        case errors::ERR_MEM_ALLOC_FAILED:
            return "Memory allocation failed";
        case errors::ERR_DEV_WRITE_FAILED:
            return "Failed to write to device";
        case errors::ERR_DEV_READ_FAILED:
            return "Failed to read from device";
        case errors::ERR_DEV_ALREADY_OPEN:
            return "Device is already open";
        case errors::ERR_DEV_SEEK_FAILED:
            return "Device seek failed";
        default:
            return "Invalid Error Code.";          
    }
    return "Unknown Error";
}

QStringList DQWindrv::getDriveNames() const {
    QStringList fd = getFloppyDrives();

    return fd;
}

QStringList DQWindrv::getDescriptiveDriveNames() const {
    QStringList descName;
    QStringList dn = getDriveNames();

    for (int i = 0; i < descName.size(); i++) {
        dn.append("Drive " + descName.at(i));
    }

    return dn;
}

QStringList DQWindrv::getFloppyDrives() {
        QStringList qsl;
        for (int i = 0; i < floppy_drive_count; i++)
                qsl.append(QString(floppy_drives[i]));

        if (detect_floppy_drives) {
            qsl.clear();
            for (int j = 0; j < win32_drive_count; j++) {
                    UINT dt = GetDriveTypeA(QString(win32_drives[j] + ":") \
                                    .toAscii().constData());

                    QString msg = "Drive ";
                    msg.append(win32_drives[j]);
                    msg.append(" is ");

                    if (dt == DRIVE_UNKNOWN)
                            msg.append("DRIVE_UNKNOWN");
                    else if (dt == DRIVE_NO_ROOT_DIR)
                            msg.append("DRIVE_NO_ROOT_DIR");
                    else if (dt == DRIVE_REMOVABLE) {
                            msg.append("DRIVE_REMOVABLE");
                            qsl.append(QString(win32_drives[j]) + ":");
                    }
                    else if (dt == DRIVE_FIXED)
                            msg.append("DRIVE_FIXED");
                    else if (dt == DRIVE_REMOTE)
                            msg.append("DRIVE_REMOTE");
                    else if (dt == DRIVE_CDROM)
                            msg.append("DRIVE_CDROM");
                    else if (dt == DRIVE_RAMDISK)
                            msg.append("DRIVE_RAMDISK");
                    else
                            msg.append("unknown :(");
            }
        }

        return qsl;
}

int DQWindrv::openDrive(QString driveName) {
    qDebug() << "DQWinDrv: Opening Drive...";
    qDebug() << "DQWinDrv: Open: Checking drive is not open";
    if (isOpen()) {
        qDebug() << "DQWinDrv: Open: A drive is already open.";
        return errors::ERR_DEV_ALREADY_OPEN;
    }

    // Build the device name
    QString dn = "\\\\.\\" + driveName;

    hDrive = INVALID_HANDLE_VALUE;

    //char filename[] = "\\\\.\\A:";

    // Open the device
    qDebug() << "DQWinDrv: Open: Opening device...";
    hDrive = CreateFileA( dn.toAscii().constData(),                              /* [in ] Filename */
                         GENERIC_READ | GENERIC_WRITE,          /* [in ] Desired access mode */
                         FILE_SHARE_READ | FILE_SHARE_WRITE,    /* [in ] Share Mode */
                         NULL,                                  /* [ino] Security Attributes */
                         OPEN_EXISTING,                         /* [in ] Creation Disposition */
                         FILE_FLAG_SEQUENTIAL_SCAN |            /* [in ] Flags and Attributes */
                         FILE_FLAG_NO_BUFFERING |               /* */
                         FILE_ATTRIBUTE_SYSTEM,                 /* */
                         NULL);                                 /* [ino] Template File */
    lastWindowsError = GetLastError();

    // Make sure we managed to open the device
    if (hDrive == INVALID_HANDLE_VALUE) {
        qDebug() << "DQWinDrv: Open: ERROR: Failed to open volume" << dn;
        qDebug() << "DQWinDrv: Open: Windows Error was:" << GetWindowsErrorString(lastWindowsError);
        return errors::ERR_DEV_OPEN_FAILED;
    }
    qDebug() << "DQWinDrv: Open: Device Open.";

    // Attempt to lock the drive
    qDebug() << "DQWinDrv: Open: Attempting to lock volume for exclusive access...";
    bool ret = LockVolume(hDrive);

    if (!ret) {
        // Lock failed. Close the volume and return.
        qDebug() << "DQWinDrv: Open: ERROR: Volume Lock failed. Aborting.";
        qDebug() << "DQWinDrv: Open: Windows Error was:" << GetWindowsErrorString(GetLastError());
        CloseHandle(hDrive);
        lastWindowsError = GetLastError();
        return errors::ERR_VOL_LOCK_FAILED;
    }
    qDebug() << "DQWinDrv: Open: Volume Locked.";

    // Update the current drive name and open status
    current_drive = driveName;
    drive_open = true;
    current_sector = fsec = -1;
    track_buffer.clear();

    qDebug() << "DQWinDrv: Open: Open Complete.";

    // And we are done.
    return errors::ERR_OK;
}

int DQWindrv::closeDrive() {
    // Firstly, dismount the volume to remove it from any filesystem drivers, etc.
    bool ret = DismountVolume(hDrive);

    if (!ret) {
        // Failed to dismount the volume. We can survive this I suppose.
        qDebug() << "WARNING: Failed to dismount volume.";
        qDebug() << "DQWinDrv: Close: Windows Error was:" << GetWindowsErrorString(GetLastError());
    }

    // Unlock the volume
    ret = UnlockVolume(hDrive);

    if (!ret) {
        // Failed to unlock volume.
        qDebug() << "ERROR: Failed to unlock volume.";
        qDebug() << "DQWinDrv: Close: Windows Error was:" << GetWindowsErrorString(GetLastError());
        return errors::ERR_VOL_UNLOCK_FAILED;
    }

    // Close the volume
    ret = CloseHandle(hDrive);
    lastWindowsError = GetLastError();

    if (!ret) {
        // Failed to close volume.
        qDebug() << "ERROR: Failed to close volume.";
        return errors::ERR_DEV_CLOSE_FAILED;
    }

    // Clear state
    current_drive = "";
    drive_open = false;
    fsec = -1;
    current_read_sector = -2;

    return errors::ERR_OK;
}

QString DQWindrv::currentDrive() const {
    return current_drive;
}

bool DQWindrv::isOpen() const {

    return drive_open;
}

bool DQWindrv::driveIsWritable(QString driveName) const {
    // We will just assume all drives are writable until a write fails.
    return true;
}

int DQWindrv::writeData(int sector, const QByteArray *data) {
    /* This function does not actually write data to disk as writing a single
       sector to disk takes about as long as writing an entire track of sectors
       to disk.

       To hugely improve performance, this function writes each sector supplied
       to a Sector Cache. When the Sector Cache contains an entire track of
       data it is flushed to disk using the write_out_track() function.

       This complicates matters slightly as we need to be able to handle
       write failures in such a way that allows the user to retry.

       Here is what must happen:
       1. If the failed sector (fsec) is >= 0 and is equal to the sector number
          for the supplied sector then:
            * The previous track failed to write
            * The user has chosen to retry
          So we should skip ahead to step 4.
       2. If the failed sector (fsec)  is >= 0 and is less than the sector number
          of the supplied sector then:
            * The previous track failed to write
            * The user has chosen to ignore the failure and move on to the next
              track.
          We must:
            * Clear the track buffer
            * Set the failed sector (fsec) to -1
       3. If the failed sector (fsec) is -1 then increment the current sector
          number and append the supplied sector to the track buffer
       4. If the track buffer contains an entire track of data, write it out to
          disk
       --- Step 5 lives in initiate_track_write
       5. If the track write succeeded:
            * Clear the track buffer
            * Set the failed sector (fsec) to -1
          If it failed:
            * Update the failed sector (fsec) to be the current sector (the
              final one in the track)
            * Return the error
     */

    // STEP 1:
    if (fsec >= 0 && fsec == current_sector) {
        // We are retrying a previously failed track write (step 4)
        // TODO: Seek back to the start of the track
        return initiate_track_write();
    }

    // STEP 2:
    // (current_sector+1) doesn't actually get updated unless it is appended to
    // the track buffer
    if (fsec >= 0 && fsec < current_sector+1) {
        // Reset track buffer
        track_buffer.clear();
        fsec = -1;
    }

    // STEP 3:
    if (fsec == -1) {
        // Append data to track buffer
        current_sector++;
        track_buffer.append(*data);
        //track_buffer.append(data);
    }

    // STEP 4:
    // Figure out how much data should be in the track buffer.
    int sectors_per_track = diskgeom.sectors;
    int bytes_per_sector = diskgeom.sectorSize;
    int bytes_per_track = sectors_per_track * bytes_per_sector;

    if (track_buffer.size() == bytes_per_track) {
        // Track buffer contains a full track of data. Write it out to disk.
        return initiate_track_write();
    }

    return errors::ERR_OK;
}

int DQWindrv::initiate_track_write() {
/* continued from writeData()

       5. If the track write succeeded:
            * Clear the track buffer
            * Set the failed sector (fsec) to -1
          If it failed:
            * Update the failed sector (fsec) to be the current sector (the
              final one in the track)
            * Return the error
*/
    // STEP 4:
    // Write out the track
    int retcode = write_out_track();

    // If it succeeded, reset the track buffer
    if (retcode == errors::ERR_OK) {
        track_buffer.clear();
        fsec = -1;
    } else {
        // The write failed.
        fsec = current_sector;
    }

    // Return the status
    return retcode;
}

int DQWindrv::write_out_track() {
    // Ok. We have a track_buffer containing all the data we should write out
    // to disk.

    // Firstly, if we havent already we must do some geometry translation
    // stuff. Windows won't understand the DIMG DiskGeometry structure
    // or its media type information so we must translate it into something
    // it can understand.
    if (pdg == 0) {
        pdg = new DISK_GEOMETRY;
        TranslateDiskGeometry(diskgeom,pdg);
    }

    DWORD dwVirtBufSize, dwBytesWritten;

    dwVirtBufSize = diskgeom.sectors * diskgeom.sectorSize;

    // Allocate a buffer.

    void* IoBuffer = VirtualAlloc(NULL,dwVirtBufSize, MEM_COMMIT, PAGE_READWRITE);

    if (!IoBuffer) {
        qDebug() << "ERROR: Unable to allocate memory for write buffer.";
        return errors::ERR_MEM_ALLOC_FAILED;
    }

    // Copy data into IoBuffer
    memcpy(IoBuffer,track_buffer.constData(),track_buffer.size());
    //for (int i = 0; i < track_buffer.count(); i++) {
        //char v = track_buffer.at(i);
        //memcpy(IoBuffer,(LPVOID)(&v),sizeof(char));
    //}

    // Write the buffer to disk
    if (!WriteFile(hDrive,(LPVOID)IoBuffer,track_buffer.count(),&dwBytesWritten,NULL)) {
        lastWindowsError = GetLastError();
        qDebug() << "ERROR: Write to device failed.";
        qDebug() << "Windows error was:" << GetWindowsErrorString(lastWindowsError);
        return errors::ERR_DEV_WRITE_FAILED;
    }
    lastWindowsError = GetLastError();

    VirtualFree(IoBuffer,0,MEM_RELEASE);
    lastWindowsError = GetLastError();

    return errors::ERR_OK;
}

int DQWindrv::readData(int sector, QByteArray *data) {
    /* We can not read each sector as it is requested - that is far too slow.
       Instead we must read in an entire track of data, store it and pull out
       the sectors as they are requested.

       So. Here is what we do.
       1. increment current_read_sector
       2. if current_read_sector == number of sectors per track then:
            * reset current_read_sector to 0
            * read_in_track()
       3. Pull current_read_sector from tread_buffer and return it
       Now, that only handles sectors being requested _in order_ from 0 to x.
       That isn't really a major problem I suppose as DIMG only ever requests
       them in order from 0. Perhaps something for future improvement though.
     */

    if (current_read_sector == -2){
        int retcoderi = read_in_track();
        if (retcoderi != errors::ERR_OK) {
            qDebug() << "Failed to read first track from device. Error code:" << retcoderi;
            return errors::ERR_DEV_READ_FAILED;
        } else
            current_read_sector++;
    }

    // STEP 1:
    current_read_sector++;

    // STEP 2:
    if (current_read_sector == diskgeom.sectors) {
        current_read_sector = 0;
        int retcoderi = read_in_track();
        if (retcoderi != errors::ERR_OK) {
            qDebug() << "Reading in track failed. Error code:" << retcoderi;
            return errors::ERR_DEV_READ_FAILED;
        }
    }

    // STEP 3:
    int readpos = current_read_sector * diskgeom.sectorSize;
    data->append(tread_buffer.mid(readpos,diskgeom.sectorSize));

    return errors::ERR_OK;
}

int DQWindrv::read_in_track() {
    qDebug() << "Reading next track...";
    tread_buffer.clear();

    // We need a BUFFER!@!!!~1
    qDebug() << "VirtualAlloc buffer...";
    DWORD dwVirtBufSize = diskgeom.sectorSize * diskgeom.sectors;
    LPVOID IoBuffer = VirtualAlloc(NULL,dwVirtBufSize, MEM_COMMIT, PAGE_READWRITE);

    if (!IoBuffer) {
        qDebug() << "ERROR: Unable to allocate memory for read buffer.";
        return errors::ERR_MEM_ALLOC_FAILED;
    }

    // Read in the track
    qDebug() << "Reading...";
    DWORD dwBytesRead;
    if (!ReadFile(  hDrive,           /* Device Handle */
                    IoBuffer,         /* Read buffer */
                    dwVirtBufSize,    /* Buffer Length */
                    &dwBytesRead,     /* Number of bytes read */
                    NULL)){           /* Overlapped */
        lastWindowsError = GetLastError();
        qDebug() << "ERROR: Read from device failed. Unable to get next track.";
        qDebug() << "Windows error was:" << GetWindowsErrorString(lastWindowsError);
        return errors::ERR_DEV_READ_FAILED;
    }
    lastWindowsError = GetLastError();
    qDebug() << "Eh? what?";

    // Convert it to a QByteArray
    tread_buffer.append((LPCSTR)IoBuffer,dwVirtBufSize);

    // Release the memory allocated for the buffer
    VirtualFree(IoBuffer,0,MEM_RELEASE);
    lastWindowsError = GetLastError();

    // And we are done.
    qDebug() << "Returning all-OK";
    return errors::ERR_OK;
}

DiskGeometry DQWindrv::getDiskGeom() {
    // Cant get the disk geometry if there is no drive open...
    if (!isOpen()) return DiskGeometry();

    /* Ok. Here is what we must do:
       1. Read in the first 1KB or so of data from the floppy drive
       2. Pass it off to setDiskGeom()
       3. Seek back to the start of the disk
     */

    qDebug() << "Attempting to guess disk geometry from first track of device...";

    // Allocate memory for reading
    LPVOID IoBuffer = VirtualAlloc(NULL, 1024, MEM_COMMIT, PAGE_READWRITE);

    if (!IoBuffer) {
        qDebug() << "Memory Allocation Failed";
        lastError =  errors::ERR_MEM_ALLOC_FAILED;
        return DiskGeometry();
    }

    // Read in the track
    DWORD dwBytesRead;
    if (!ReadFile(  hDrive,           /* Device Handle */
                    (LPVOID)IoBuffer, /* Read buffer */
                    1024,             /* Buffer Length */
                    &dwBytesRead,     /* Number of bytes read */
                    NULL)){           /* Overlapped */
        lastWindowsError = GetLastError();
        qDebug() << "ERROR: Read from device failed. Unable to get disk geometry.";
        qDebug() << "Windows error was:" << GetWindowsErrorString(lastWindowsError);
        lastError = errors::ERR_DEV_READ_FAILED;
        return DiskGeometry();
    }


    // Seek back to the start of the disk (so it is ready for reading later)
    DWORD newpos =
         SetFilePointer(hDrive,     /* Device Handle */
                       0,           /* distance to move (lower 32 bits) */
                       NULL,        /* Upper 32bits of distance to move */
                       FILE_BEGIN); /* Move Method (where to start from) */
    if (newpos != 0) {
        lastWindowsError = GetLastError();
        qDebug() << "ERROR: Failed to seek to start of device";
        qDebug() << "Windows error was:" << GetWindowsErrorString(lastWindowsError);
        lastError =  errors::ERR_DEV_SEEK_FAILED;
        return DiskGeometry();
    }

    // Convert IoBuffer to a QByteArray
    QByteArray iob;
    iob.append((LPCSTR)IoBuffer,dwBytesRead);

    // Pass the IoBuffer off to setDiskGeom()
    setDiskGeom(&iob);

    // free any allocated memory
    VirtualFree(IoBuffer,0,MEM_RELEASE);

    return diskgeom;
}

int DQWindrv::setDiskGeom(QByteArray *boot_sector){
    /* Try to detect a DOS Disk Geometry.
     */

    int e;

    qDebug() << "Attempting to guess disk geometry based on boot sector...";

    // Try to guess the disk geometry
    qDebug() << "Attempting DOS Geometry...";
    e = geom_detect::dosgeom(&diskgeom, boot_sector);
    if (e == geom_err::BAD_FORMAT) {
        qDebug() << "Not DOS Geometry. No other Geometries are supported. END.";
        return e;
    }

    return e;
}

int DQWindrv::setDiskGeom(DiskGeometry dg){
    diskgeom = dg;
    return errors::ERR_OK;
}

// Note: This function requires Windows 2000 or newer. It is not compatible
// with Windows 9x.
bool DQWindrv::DismountVolume(HANDLE hDisk)
{
   DWORD ReturnedByteCount;

   int ret = DeviceIoControl(hDisk, /* [in ] Device Handle */
      FSCTL_DISMOUNT_VOLUME,        /* [in ] IO Control Code */
      NULL,                         /* [in ] In Buffer */
      0,                            /* [in ] In Buffer Size */
      NULL,                         /* [out] Out Buffer */
      0,                            /* [out] Out Buffer Size */
      &ReturnedByteCount,           /* [out] Bytes Returned */
      NULL);                        /* [io ] Overlapped */
   lastWindowsError = GetLastError();

   /* FSCTL_DISMOUNT_VOLUME: Dismounts a volume. The operating system will
      attempt to remount it when an attempt is made to access it.
    */

   // If DeviceIoControl returns 0 that means something went wrong. Otherwise
   // it returned a non-zero value which means success.
   return (ret == 0)? false : true;
}

// Note: This function requires Windows 2000 or newer. It is not compatible
// with Windows 9x.
bool DQWindrv::LockVolume(HANDLE hDisk)
{
   DWORD ReturnedByteCount;

   int ret = DeviceIoControl(hDisk, /* [in ] Device Handle */
      FSCTL_LOCK_VOLUME,            /* [in ] IO Control Code */
      NULL,                         /* [in ] In Buffer */
      0,                            /* [in ] In Buffer Size */
      NULL,                         /* [out] Out Buffer */
      0,                            /* [out] Out Buffer Size */
      &ReturnedByteCount,           /* [out] Bytes Returned */
      NULL);                        /* [io ] Overlapped */
   lastWindowsError = GetLastError();

   /* FSCTL_LOCK_VOLUME: Locks a volume. A locked volume can be accessed only
      through handles to the file object (*hDevice) that locks the volume.
    */

   // If DeviceIoControl returns 0 that means something went wrong. Otherwise
   // it returned a non-zero value which means success.
   return (ret == 0)? false : true;
}

// Note: This function requires Windows 2000 or newer. It is not compatible
// with Windows 9x.
bool DQWindrv::UnlockVolume(HANDLE hDisk) {
   DWORD ReturnedByteCount;

   int ret = DeviceIoControl(hDisk, /* [in ] Device Handle */
      FSCTL_UNLOCK_VOLUME,          /* [in ] IO Control Code */
      NULL,                         /* [in ] In Buffer */
      0,                            /* [in ] In Buffer Size */
      NULL,                         /* [out] Out Buffer */
      0,                            /* [out] Out Buffer Size */
      &ReturnedByteCount,           /* [out] Bytes Returned */
      NULL);                        /* [io ] Overlapped */
   lastWindowsError = GetLastError();

   /* FSCTL_UNLOCK_VOLUME: Unlocks a volume
    */

   // If DeviceIoControl returns 0 that means something went wrong. Otherwise
   // it returned a non-zero value which means success.
   return (ret == 0)? false : true;
}

// Note: This function requires Windows 2000 or newer. It is not compatible
// with Windows 9x.
bool DQWindrv::GetDiskGeometry(HANDLE hDisk, PDISK_GEOMETRY lpGeometry)
{
   DWORD ReturnedByteCount;


   /*int ret =*/ DeviceIoControl(hDisk,     /* [in ] Device Handle */
      IOCTL_STORAGE_GET_MEDIA_TYPES,    /* [in ] IO Control Code */
      NULL,                             /* [in ] In Buffer */
      0,                                /* [in ] In Buffer Size */
      lpGeometry,                       /* [out] Out Buffer */
      sizeof(lpGeometry),               /* [out] Out Buffer Size */
      &ReturnedByteCount,               /* [out] Bytes Returned */
      NULL);                            /* [io ] Overlapped */
   lastWindowsError = GetLastError();

   /* IOCTL_STORAGE_GET_MEDIA_TYPES: Retrieves the geometry information for the
      device.
    */

   // If DeviceIoControl returns 0 that means something went wrong. Otherwise
   // it returned a non-zero value which means success.
   //return (ret == 0)? false : true;
   return false;
}

// Requires Win2k or newer.
int DQWindrv::GetFloppyDriveType(HANDLE hDisk) {

   DWORD ReturnedByteCount;
   DISK_GEOMETRY Geoms[20];

   int ret = DeviceIoControl(hDisk,     /* [in ] Device Handle */
      IOCTL_STORAGE_GET_MEDIA_TYPES,    /* [in ] IO Control Code */
      NULL,                             /* [in ] In Buffer */
      0,                                /* [in ] In Buffer Size */
      Geoms,                            /* [out] Out Buffer */
      sizeof(Geoms),                    /* [out] Out Buffer Size */
      &ReturnedByteCount,               /* [out] Bytes Returned */
      NULL);                            /* [io ] Overlapped */
   lastWindowsError = GetLastError();

   if (ret > 0 && ReturnedByteCount > 0) {
        switch (Geoms[0].MediaType){
            case F5_1Pt2_512: // 5.25 1.2MB floppy
            case F5_360_512:  // 5.25 360K  floppy
            case F5_320_512:  // 5.25 320K  floppy
            case F5_320_1024: // 5.25 320K  floppy
            case F5_180_512:  // 5.25 180K  floppy
            case F5_160_512:  // 5.25 160K  floppy
              return 525;

            case F3_1Pt44_512: // 3.5 1.44MB floppy
            case F3_2Pt88_512: // 3.5 2.88MB floppy
            case F3_20Pt8_512: // 3.5 20.8MB floppy
            case F3_720_512:   // 3.5 720K   floppy
              return 350;

            case RemovableMedia:
              return 2;

            case FixedMedia:
              return 1;

            default:
              return 0;
        }
   }
   return 0;
}

MEDIA_TYPE DQWindrv::GetMediaType(DiskGeometry dg) {
// 		RATE_HD,	/* Data rate for 1.4Mb 3.5"  in 3.5"  drive */
//		RATE_DD,	/* Data rate for 360k  5.25" in 1.2Mb drive */
//		RATE_SD,	/* Data rate for 720k  3.5"  in 3.5"  drive */
//		RATE_ED		/* Data rate for 2.8Mb 3.5"  in 3.5"  drive */

    switch (dg.dataRate) {
        case GeomStr::RATE_HD:
            // Data rate for 1.4Mb 3.5"  in 3.5"  drive or 1.2Mb 5.25" in 5.25" drive

            // This could be either a 5.25" disk or a 3.5" disk. We will just
            // look at the target drive type and take a guess.
            if (GetFloppyDriveType(hDrive) == 525)
                return F5_1Pt2_512;
            else
                return F3_1Pt44_512;
        case GeomStr::RATE_DD:
            // Data rate for 360k  5.25" in 1.2Mb drive
            return F5_360_512;
        case GeomStr::RATE_SD:
            // Data rate for 720k  3.5"  in 3.5"  drive
            return F3_720_512;
        case GeomStr::RATE_ED:
            // Data rate for 2.8Mb 3.5"  in 3.5"  drive
            return F3_2Pt88_512;
        default:
            // Just assume it is a 1.44MB disk in a 3.5" drive - that is what
            // most of them will be anyway.
            return F3_1Pt44_512;
    }

}

QString DQWindrv::GetWindowsErrorString(unsigned long error) {
    // The message buffer. This will be created by FormatMessage()
    unsigned short *lpMsgBuf;

    // Pull the error message out of windows. This will, at the same time,
    // create the message buffer for us as FORMAT_MESSAGE_ALLOCATE_BUFFER
    // has been specified.
    int stlen = FormatMessage(
        FORMAT_MESSAGE_ALLOCATE_BUFFER |            /* Flags */
        FORMAT_MESSAGE_FROM_SYSTEM |                /* */
        FORMAT_MESSAGE_IGNORE_INSERTS,              /* */
        NULL,                                       /* Source (optional) */
        error,                                      /* Message ID */
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),  /* Language ID */
        (LPTSTR) &lpMsgBuf,                         /* Message Buffer */
        0,                                          /* Buffer Size */
        NULL );                                     /* Arguments (optional) */

    // Convert the message into a QString
    QString error_message = QString::fromUtf16(lpMsgBuf,stlen);

    // Free up the buffer
    LocalFree(lpMsgBuf);

    // Return the result.
    return error_message;
}

int DQWindrv::TranslateDiskGeometry(DiskGeometry dg, PDISK_GEOMETRY lpGeometry) {
    if (lpGeometry == 0) return errors::ERR_NULL_POINTER;

    lpGeometry->BytesPerSector = dg.sectorSize;
    lpGeometry->Cylinders.LowPart = dg.cylinders;   // This is a weird 64bit number thing
    lpGeometry->MediaType = GetMediaType(dg);
    lpGeometry->SectorsPerTrack = dg.sectors;
    lpGeometry->TracksPerCylinder = dg.heads;

    return errors::ERR_OK;
}

Q_EXPORT_PLUGIN2(dqwindadrv, DQWindrv)
