/**
    This file is part of "DIMG".

    This is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This software is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this software; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
-----------------------------------------------------
DIMG Class `Imager'
-----------------------------------------------------
  Purpose: To handle reading and writing disk images through the use of
           DQDrivers.
 */

#ifndef IMAGER_H
#define IMAGER_H

#include <QObject>
#include <QApplication>
#include <QFileDialog>
#include <QPluginLoader>
#include <QFileInfo>
#include <QMessageBox>
#include <QtDebug>
#include <QProgressDialog>
#include <QSettings>

// Drive Driver interface.
#include <DQDriver.h>
#include <DiskGeometry.h>

#include "errors.h"

/** This class handles disk imaging tasks. It is in charge of interacting with
  the Drive Access Plugins.
  */
class Imager: public QObject
{
    Q_OBJECT

public:
    explicit Imager(QObject *parent);
    virtual ~Imager();

    /** Reads an image from the currently selected drive and stores it in the destination device.
      The destination device is expected to be Closed at the start of the operation and will be
      closed upon the completion of the operation.

      \param destination_device The device to write the image to.
      \returns An error code.
     */
    int read_image(QIODevice * destination_device);

    /** Writes the supplied image to the currently selected drive. The source device is expected to
      be closed and will be closed upon the completion of this operation.

      \param source_device The device containing the image to be written.
      \returns An error code.
     */
    int write_image(QIODevice * source_device);

    /** Loads the Drive Access Plugins

      \returns An error code.
     */
    int load_drivers();

    /** Returns a list of available drive names. These are to be used when setting the current drive.

      An example of a drive name may be "A:" or "VIRTFD::"

      \returns A list containing the names of all available drives.
     */
    QStringList getDriveNames();

    /** Gets the descriptive name for each available drive. The ordering of this list is the same as
      the list returned by getDriveNames(). That is, item 0 in that list will match up with item 0
      in this list.

      An example of a descriptive drive name may be "A:" or "Virtual Floppy drive".

      \returns A list containing descriptive names for all available drives.
     */
    QStringList getDescriptiveDriveNames();

    /** Gets the currently selected drive name.

      \returns The currently selected drive name.
     */
    QString get_current_drive();

    /** Returns true if the current Drive Access Plugin is ready.
      \returns True if the current Drive Access Plugin is ready.
     */
    bool driver_ready();

    /** Gets the name of the current drive access plugin.
      \returns The name of the current drive access plugin.
     */
    QString get_driver_name();

    /** Gets the version string for the current drive access plugin.
      \returns The name of the current drive access plugin.
     */
    QString get_driver_version();

    /** Sets the parent widget to be used for progress dialogs, etc. If not
      set the progress dialogs will not be modal.

      \param parent The parent widget for progress dialogs, etc.
     */
    void set_parent_widget(QWidget *parent) { pw = parent;}

    /** Returns true if the DQDskDriver plugin is configurable
      \returns True if the DQDskDriver is configurable
     */
    bool configurable();

public slots:
    /** Sets the currently active drive to the one specified. The supplied
      drive name must be one of those returned by getDriveNames().

      \param driveName The drive to set as active.
     */
    void set_current_drive(QString driveName);

    /** If the DQDskDriver is configurable, this will cause it to be configured. If
      it is not configurable this will do nothing.
     */
    void configure();

private:
    bool loadDQPlugin();    /*!< Loads the DQDriver Plugin */

    int open_current_drive();       /*!< Attempts to opent the current drive */

    QString currentDrive;       /*!< The current disk drive */

    DQDskDriver *dqdrv;     /*!< The current disk driver */

    QWidget *pw; /*!<Parent Widget for progress dialogs, etc. */
};

#endif // IMAGER_H
