#ifndef FSENTRY_H
#define FSENTRY_H

#include <QDateTime>
#include <QIcon>
#include <QString>
#include <QList>
#include "FSPlugin.h"

class FSEntry
{
public:

    /** This enumeration specifies to the getDate() and getTime() which field
        should be returned. Only fields that are indicated as being supported
        by the FSPlugins getFeatures() method should be used.
     */
    enum DateType {
        D_CreatedDate,
        D_CreatedTime,
        D_AccessedDate,
        D_AccessedTime,
        D_ModifiedDate,
        D_ModifiedTime
    };

    virtual ~FSEntry() {}  /*!< Does nothing */

    /** Returns this entries attributes.

      The attributes are defined in FSPlugin as a typedef for QFlags<Attribute>.

      \returns This entries attributes.
     */
    virtual FSPlugin::Attributes getAttributes() const = 0;

    /** Convenience function to see if this entry represents a directory. The
        same information can be obtained by looking at the value returned by
        getAttributes().

        \returns True if this entry represents a directory.
     */
    virtual bool isDirectory() const = 0;

    /** Returns the size of this entry. It has no meaning for directories so
      should return 0 if it is a directory.

      If there is any error, 0 should be returned.

      \returns The size of the file in Bytes.
     */
    virtual quint64 getSize() const = 0;

    /** Returns the base name of the file without the path or extension.

      The base name contains all characters in the filename up to (but not
      including) the first '.' character.

      \returns The base name of the entry.
     */
    virtual QString baseName() const = 0;

    /** Returns the filenames suffix. That is, every character after the first
      '.' character in the filename.

      \returns The suffix of the entry.
     */
    virtual QString suffix() const = 0;

    /** The full FileName of the entry.

      \returns The full filename of the entry.
     */
    virtual QString fileName() const = 0;

    /** Returns the file path.

      \Returns The file path.
     */
    virtual QString path() const = 0;

    /** Returns the file name, including the path.

      \returns The file name including the path.
     */
    virtual QString filePath() const = 0;

    /** Gets the specified Date value for the file.

      Only date values that the FSPlugins getFeatures() method has indicated
      as being supported should be requested here.

      The mapping is:
        - D_CreatedDate support is indicated by F_CreationDate
        - D_AccessedDate support is indicated by F_AccessDate
        - D_ModifiedDate support is indicated by F_ModDate


      \param dt The date type to return
      \returns The value for the specified DateType. If the specified DateType
            is for a time, QDate() will be returned.
     */
    virtual QDate getDate(DateType dt) const = 0;

    /** Gets the specified Time value for the file.

      Only date values that the FSPlugins getFeatures() method has indicated
      as being supported should be requested here.

      The mapping is:
        - D_CreatedTime support is indicated by F_CreationTime
        - D_AccessedTime support is indicated by F_AccessTime
        - D_ModifiedTime support is indicated by F_ModTime


      \param dt The time type to return
      \returns The value for the specified DateType. If the specified DateType
            is for a date, QTime() will be returned.
     */
    virtual QTime getTime(DateType dt) const = 0;

    /** Gets an icon that represents this files type for display purposes. How
        exactly this should be interpreted depends on the filesystems native
        platform. For example, FAT filesystems might be able to use
        QFileIconProvider to provide platform-native icons for a files
        extension.

      \returns An icon that represents this files type.
     */
    virtual QIcon getIcon() const = 0;
};

#endif // FSENTRY_H
