#ifndef UTILS_H
#define UTILS_H

#include <QByteArray>

/** A collection of utility functions.
 */
namespace utils {
    /** Swaps the byte ordering of the supplied QByteArray and returns it.

      \param b The array of bytes to swap the ordering of
      \returns The input data with its byte ordering swapped.
     */
    QByteArray swapEndian(QByteArray b);

    /** Swaps the byte ordering of the input QByteArray and then converts it
      to a 32bit integer.
      \param b The data to convert to an integer. The conversion starts at
                byte 0 and runs for four bytes. For the conversion to be
                successfull, there must be at least four bytes in the array.
      \returns The 32bit integer representation of the input data.
     */
    int byteArrayToIntS(QByteArray b);

    /** Converts the input byte array to a 32bit integer.

      At least four bytes are required to perform the conversion. The
      conversion starts at byte 0.

      It is a convenience function that just calls
      byteArrayToInt(QByteArray,int) passing it the input data and an offset
      of 0.

      \param b The byte array to convert to an integer.
      \returns The 32bit integer representation of the input data.
     */
    int byteArrayToInt(QByteArray b);

    /** Converts the input byte array to a 32bit integer starting at the
      specified offset.

      \param b The byte array to convert
      \param offset The place in the array to start converting. This offset
                    must be at least four bytes before the end of the array.
      \returns A 32bit integer representation of the four bytes starting at the
               specified offset.
     */
    int byteArrayToInt(QByteArray b, int offset);

    /** Returns true if the specified bit is set in the supplied byte of data.

      \param byte The byte to get the status of a bit from
      \param bit The bit number to be checked.
      \returns True if the specified bit is set in the supplied byte.
     */
    bool getBit(char byte, int bit);

    /** Returns a series of bytes from the input QByteArray.

      This method will return a QByteArray containing the exact number of bytes
      specified by the length parameter. If the input QByteArray or range is
      smaller than the specified length, the output will be padded with null
      bytes at the end.

      \param data The source data to copy into the output.
      \param start The location in the input data array to start copying from.
      \param end The location to the input data array to stop copying and start
                 padding with null bytes. If the difference between start and
                 end is the same as length, no null-byte padding will occur.
      \param length The number of bytes to copy. If there are not enough bytes
                    in the specified range, the output will be padded with
                    null-bytes.
      \returns All bytes between start and end, padded out to length with null
                bytes if required.
     */
    QByteArray getBytes(QByteArray data, int start, int end, int length);

    /** Copies the specified range of bytes from the input data array to the
        output.

        The end location should not be beyond the end of the input QByteArray.

        \param data The data array to copy from
        \param start The location in the input data array to start copying from
        \param end The location in the input data array to stop copying.
        \returns All data between the start and end locations in the input data
                 array.
     */
    QByteArray getBytes(QByteArray data, int start, int end);
}
#endif // UTILS_H
