#ifndef FAT12FS_H
#define FAT12FS_H

// Qt headers
#include <QIODevice>
#include <QString>
#include <QList>
#include <QFlags>
#include <QtPlugin>

// Filesystem plugin header
#include <FSPlugin.h>

// Fat12fs headers
#include "fat12.h"

#include "about_dialog.h"

class Fat12fs : public QObject, public FSPlugin
{
    Q_OBJECT
    Q_INTERFACES(FSPlugin)

public:
    Fat12fs() { fs = 0; }
    ~Fat12fs() {if (fs != 0) delete fs;}

    /** This function is called to load the filesystem from the
        specified device. When this function returns, the filesystem
        should be ready to use.

        If a filesystem is already open, this should return False.

        This virtual function must be implemented by all subclasses.

        \param dev This is a pointer to the QIODevice that the filesystem will
                    work off. Only Random Access devices are supported.
        \returns True if the filesystem was able to load successfully. If there
                 was an error, False should be returned.
    */
    bool load(QIODevice * dev);

    /** This function is called when the filesystem should be closed. This must
        be called before the filesystem can be loaded from another QIODevice.

        This virtual function must be implemented by all subclasses.

        \returns True if filesystem could be closed. If there were any errors,
                 this function should return False.
     */
    bool close();

    bool isClosed() { return is_closed; }

    /** This method should return a version string for this plugin. Something
        like "1.9.0-beta3" would be acceptable. It is only used for display
        purposes.

        This virtual function must be implemented by all subclasses.

        \returns A string containing the version of this plugin.
     */
    QString getVersionString() const {return "1.0.1";}

    /** This method should return the name of the plugin. Something like
        "FAT12 Filesystem" would be acceptable. It is only used for display
        purposes.

        When combined with getVersionString(), the output would look something
        like:
           FAT12 Filesystem version 1.9.0-beta3

        \returns A string containing the name of this plugin.
     */
    QString getPluginName() const {return "fat12fs";}

    /** This function indicates which features the Filesystem supports.
     * The information it returns is used to decide which features the
     * plugin supports. For example, if the F_DisplayAbout flag is not set
     * then the DisplayAbout method will never be called.
     */
    Features getFeatures() const;

    /** This function indicates which file attributes the Filesystem supports.
      No data structure ever returned by a subclass should make use of
      attributes not specified in the value this function returns.

      If a data structure returned by a subclass uses an attribute that is not
      returned by this method, it is a bug.

      \returns All attributes returned by this filesystem (Attributes that are
               only used internally dont need to be returned here).
     */
    Attributes getSupportedAttributes() const;

    /** Returns the filesystems Volume Label if supported by the filesystem.
        This method will only be called if the F_VolumeLabel flag is returned
        by the getSupportedAttributes() method.

        \returns The filesystems Volume Label.
     */
    QString volumeLabel() const;

    /** Returns the error-code for the last error that occured.
      \returns The error-code for the last error that occured.
     */
    int getLastErrorNumber() const;

    /** Returns the error string for the last error that occured.
      \returns The error string for the last error that occured.
     */
    QString getLastErrorString() const;

    /** Copies a file from this filesystem into the computers filesystem.

        For example, if a subclass implemented this interface over a zip file,
        this method would be the extract operation. When called, it would
        extract the specified file from the zip file to the specified location
        on a mounted disk on the local system.

        \param source_filename The name of the file in this filesystem to copy.
                    It takes any filename that is valid for this filesystem
        \param destination_filename The name of the destination file that the
                 source file should be copied to. It takes any filename that
                 is valid for the local system.
        \returns True if the copy operation was successfull, False if there was
                 an error.
     */
    bool copyFile(QString source_filename, QString destination_filename);

    /** Reads a file into memory and returns it.

        Because this method loads the entire file into memory, it should only
        be used for small files. A Read method may be provided at some point in
        the future when more complex filesystems arise using this interface.

        \param filename
     */
    QByteArray readFile(QString filename);

    /** Returns an FSEntry that represents the specified file or directory.

      The FSEntry returned by this function should represent the specified
      filename. If the specified file could not be found, it should just return
      null.

      It is the responsibility of the caller to delete the FSEntry returned
      by this method. The FSPlugin will not keep track of it.

      \param filename The name of the file to get the FSEntry for.
      \returns The FSEntry that represents the specified file or directory.
     */
    FSEntry * getEntryByName(QString filename);

    /** Returns a list of all FSEntries in the specified directory. If the
      directory could not be found, it should return an empty list.

      It is the responsibility of the caller to delete every FSEntry returned
      by this method. The FSPlugin will not keep track of them.

      \param directory The name of the directory to get an entry listing for.
      \returns An entry listing for the specified directory.
     */
    QList<FSEntry *> getEntryList(QString directory);

    /** This should display or output some form of about screen for this
        plugin.

        The default implementation does nothing - if a subclass does not
        reimplement it, this should indicated by the flags returned by
        the getFeatures() method.

        This function will only ever be called if getFeatures() indicates that
        it has been implemented.
     */
    void displayAbout() const;

private:
    FAT12 *fs;
    bool is_closed;
};

#endif // FAT12FS_H
