#ifndef FAT12ENTRY_H
#define FAT12ENTRY_H

// Qt headers
#include <QFileIconProvider>

// FSPlugin headers
#include <fsentry.h>

// Fat12fs headers
#include "fat12fs.h"
#include "fat12_direntry.h"
#include "winutils.h"

class Fat12Entry : public FSEntry
{
public:
    ~Fat12Entry() {}

    /** Constructs a new Fat12Entry from the supplied FAT12_DirEntry.

      \param e The FAT12_DirEntry to construct this entry from
      \param directory The path for this entry. Eg, if the full filename is
               /tmp/file.txt, the directory would be /tmp
     */
    Fat12Entry(FAT12_DirEntry e, QString directory);

    /** Returns this entries attributes.

      The attributes are defined in FSPlugin as a typedef for QFlags<Attribute>.

      \returns This entries attributes.
     */
    FSPlugin::Attributes getAttributes() const;

    /** Convenience function to see if this entry represents a directory. The
        same information can be obtained by looking at the value returned by
        getAttributes().

        \returns True if this entry represents a directory.
     */
    bool isDirectory() const;

    /** Returns the size of this entry. It has no meaning for directories so
      should return 0 if it is a directory.

      If there is any error, 0 should be returned.

      \returns The size of the file in Bytes.
     */
    quint64 getSize() const;

    /** Returns the base name of the file without the path or extension.

      The base name contains all characters in the filename up to (but not
      including) the first '.' character.

      \returns The base name of the entry.
     */
    QString baseName() const;

    /** Returns the filenames suffix. That is, every character after the first
      '.' character in the filename.

      \returns The suffix of the entry.
     */
    QString suffix() const;

    /** The full FileName of the entry.

      \returns The full filename of the entry.
     */
    QString fileName() const;

    /** Returns the file name, including the path.

      \returns The file name including the path.
     */
    QString filePath() const;

    /** Returns the file path.

      \Returns The file path.
     */
    QString path() const;

    /** Gets the specified Date value for the file.

      Only date values that the FSPlugins getFeatures() method has indicated
      as being supported should be requested here.

      The mapping is:
        - D_CreatedDate support is indicated by F_CreationDate
        - D_AccessedDate support is indicated by F_AccessDate
        - D_ModifiedDate support is indicated by F_ModDate


      \param dt The date type to return
      \returns The value for the specified DateType. If the specified DateType
            is for a time, QDate() will be returned.
     */
    QDate getDate(DateType dt) const;

    /** Gets the specified Time value for the file.

      Only date values that the FSPlugins getFeatures() method has indicated
      as being supported should be requested here.

      The mapping is:
        - D_CreatedTime support is indicated by F_CreationTime
        - D_AccessedTime support is indicated by F_AccessTime
        - D_ModifiedTime support is indicated by F_ModTime


      \param dt The time type to return
      \returns The value for the specified DateType. If the specified DateType
            is for a date, QTime() will be returned.
     */
    QTime getTime(DateType dt) const;

    /** Gets an icon that represents this files type for display purposes. How
        exactly this should be interpreted depends on the filesystems native
        platform. For example, FAT filesystems might be able to use
        QFileIconProvider to provide platform-native icons for a files
        extension.

      \returns An icon that represents this files type.
     */
    QIcon getIcon() const;
private:
    FAT12_DirEntry ent;
    QString entry_path;
};

#endif // FAT12ENTRY_H
