#include "fat12_direntry.h"

FAT12_DirEntry::FAT12_DirEntry()
{
}

FAT12_DirEntry::FAT12_DirEntry(QByteArray data){
    load_entry(data);
}

void FAT12_DirEntry::load_entry(QByteArray data) {
    /* Ok, We need to do the following:
       1) Split up the input data. In order of appearance, this is:
            FileName, Attributes, Reserved, EA Data, date/time stamp,
            Entry Cluster, File Size
       2) Then we need to convert strings, etc to the right format
       3) Then we need to decode the MS-DOS time/date stamp
       4) Lastly, we need to decode the attributes.
     */

    // Store the data for future use
    entry_data = data;

    // The FileName - bytes 0-7
    QByteArray t_filename;
    for (int i = 0; i <= 7; i++) {
        t_filename.append(data.at(i));
    }

    // The file extension - 3 characters, 3 bytes from 8 to 10
    QByteArray t_extension;
    for (int i = 8; i <= 10; i++) {
        t_extension.append(data.at(i));
    }

    // Attributes - byte 11
    char t_attributes = data.at(11);


    // Reserved - two bits are used by NT and later versions to encode case
    // information. Otherwise, it should be 0.
    //char c_res_1 = data.at(12);

    // Created time, fine resolution - 10ms units from 0 to 199
    //char ct_ms = data.at(13);

    // Created time - hour, minute and second.
    QByteArray ct_time;
    ct_time.append(data.at(14));
    ct_time.append(data.at(15));
    create_time = convertTime(data.at(14),data.at(15));

    // Created date - year, month and day
    QByteArray ct_date;
    ct_date.append(data.at(16));
    ct_date.append(data.at(17));
    create_date = convertDate(data.at(16),data.at(17));

    // Last access date - year, month and day
    QByteArray at_date;
    at_date.append(data.at(18));
    at_date.append(data.at(19));
    access_date = convertDate(data.at(18), data.at(19));

    // Extended Attribute Index (used by OS/2 and NT). In FAT32 this is the
    // high 2 bytes of the first cluster number.
    QByteArray ea_data;
    ea_data.append(data.at(20));
    ea_data.append(data.at(21));

    // Last Modified time.
    QByteArray mt_time;
    mt_time.append(data.at(22));
    mt_time.append(data.at(23));
    modified_time = convertTime(data.at(22),data.at(23));

    // Last Modified date.
    QByteArray mt_date;
    mt_date.append(data.at(24));
    mt_date.append(data.at(25));
    modified_date = convertDate(data.at(24),data.at(25));

    // Entry cluster - bytes 26 and 27.
    // Now, this is a 32bit integer stored little-endian.
    QByteArray t_entry_cluster;
    t_entry_cluster.append(data.at(26));
    t_entry_cluster.append(data.at(27));
    t_entry_cluster.append('\0');
    t_entry_cluster.append('\0');

    // File size - bytes 28 to 31
    QByteArray t_file_size;
    for (int i = 28; i <= 31; i++) {
        t_file_size.append(data.at(i));
    }

    // Ok, we have all the data. Now time to convert it!
    m_filename = QString(t_filename);
    ext = QString(t_extension);

    loadAttributes(t_attributes);

    // Get the integers
    entryCluster = utils::byteArrayToIntS(t_entry_cluster);
    fileSize = utils::byteArrayToIntS(t_file_size);
}

QString FAT12_DirEntry::fullFileName() const {
    if (ext.trimmed().isEmpty())
        return m_filename.trimmed();
    else
        return m_filename.trimmed() + "." + ext.trimmed();
}

void FAT12_DirEntry::loadAttributes(char attributes) {
    this->attributes = attributes;

    read_only = utils::getBit(attributes,0);
    hidden = utils::getBit(attributes,1);
    system_file = utils::getBit(attributes,2);
    volume_id = utils::getBit(attributes,3);
    directory = utils::getBit(attributes,4);
    archive = utils::getBit(attributes,5);
}

FAT12_DirEntry FAT12_DirEntry::copy() {
    return FAT12_DirEntry(entry_data);
}

QTime FAT12_DirEntry::convertTime(char byteA, char byteB) {
    /* MS-DOS stores the time as follows:
         Bits      Field        Range
         0-4     Seconds/2      0-29
         5-10     Minutes       0-59
         15-11     Hours        0-23
     */

    // First up, lets get both bytes into one integer so we can move and mask
    // stuff freely.
    QByteArray qba;
    qba.append(byteA);
    qba.append(byteB);
    qba.append('\0');
    qba.append('\0');

    int data = utils::byteArrayToIntS(qba);

    // Ok, the two bytes should be at the start of the int data.

    // Now, we are after the Seconds field. This is the first 5 bits. This
    // means we need to mask everything else out.
    int mask = 0x0000001F;

    // Now we mask out the unwanted data
    int seconds = data & mask;

    // Multiply by 2 (as DOS only records every second second
    seconds = seconds * 2;
    // And we now have the number of seconds.

    // Next up, the Minutes field. First we must shift the data to the right
    // by 5 bits:
    data = data >> 5;

    // Now we want the lower 6 bits - thats 0x3F
    mask = 0x0000003F;

    // Mask out the unwanted stuff (the hours)
    int minutes = data & mask;
    // And we should have the minutes now.

    // Lastly, the Hours field. All we have to do for this is shift. No masking
    // required.
    int hours = data >> 6;

    // And now we just have to convert to QTime and we are done!
    QTime t(hours,minutes,seconds,0);

    return t;
}
QDate FAT12_DirEntry::convertDate(char byteA, char byteB) {
    /* MS-DOS stores the date as follows:
         Bits      Field        Range   Comments
         0-4        Day         1-31
         5-8       Month        1-12    1=January, 12=December
         9-15      Year         0-127   0-1980, 127=2107
     */

    // Firstly, put all the data in an int
    QByteArray qba;
    qba.append(byteA);
    qba.append(byteB);
    qba.append('\0');
    qba.append('\0');

    int data = utils::byteArrayToIntS(qba);

    // Now make a mask for the lower 5 bits
    int mask = 0x0000001F;

    // Pull out the day
    int day = data & mask;

    // Shift the day data out of the way
    data = data >> 5;

    // Prepare the mask to give us the month
    mask = 0x0000000F;

    // Pull out the month
    int month = data & mask;

    // Move the month out of the way to give us years
    int year = data >> 4;

    // Now, the years start at 1980 so we should just add
    // 1980 to the year we have so far:
    year += 1980;

    // And then convert it all to a QDate
    QDate d(year,month,day);

    return d;
}

QString FAT12_DirEntry::toString() {
    if (filename().isEmpty()) return "";

    // Long Filename entries have the following attributes set:
    //      Volume Label, System, Hidden, and Read Only
    // We do not currently support long filenames.
    if (volume_id && system_file && hidden && read_only)
        return "";

    QString value;

    if (!directory) {
        value.append(m_filename.trimmed());
        value.append(".");
        value.append(ext.trimmed());
    } else {
        value.append("<");
        value.append(m_filename.trimmed());
        value.append(ext.trimmed());
        value.append(">");
    }

    if (value.length() < 8) value.append("\t");
    value.append("\t");

    // Do the attributes
    if (read_only) value.append("r");
    else value.append("-");
    if (hidden) value.append("h");
    else value.append("-");
    if (system_file) value.append("s");
    else value.append("-");
    if (archive) value.append("a");
    else value.append("-");
    if (volume_id) value.append("v");
    else value.append("-");
    if (directory) value.append("d");
    else value.append("-");

    // another layout tab
    value.append("\t");

    // and now for the file size.
    QString sz = "B";
    int tfsz = fileSize;
    if (tfsz > 1024) {
        tfsz = tfsz / 1024;
        sz = "KB";
    }
    if (tfsz > 1024) {
        tfsz = tfsz / 1024;
        sz = "MB";
    }
    if (tfsz > 1024) {
        tfsz = tfsz / 1024;
        sz = "GB";
    }
    value.append(QString::number(tfsz));
    value.append(sz);

    value.append("\t");
    value.append(QString::number(entryCluster));

    return value;
}

QStringList FAT12_DirEntry::toValues() {
    QStringList qsl;

    if (filename().isEmpty())
        return qsl;

    // Long Filename entries have the following attributes set:
    //      Volume Label, System, Hidden, and Read Only
    // We do not currently support long filenames.
    if (volume_id && system_file && hidden && read_only)
        return qsl;

    QString value;

    if (!directory) {
        value.append(m_filename.trimmed());
        value.append(".");
        value.append(ext.trimmed());
    } else {
        value.append("<");
        value.append(m_filename.trimmed());
        value.append(ext.trimmed());
        value.append(">");
    }

    qsl.append(value);
    value = "";

    // Do the attributes
    if (read_only) value.append("r");
    else value.append("-");
    if (hidden) value.append("h");
    else value.append("-");
    if (system_file) value.append("s");
    else value.append("-");
    if (archive) value.append("a");
    else value.append("-");
    if (volume_id) value.append("v");
    else value.append("-");
    if (directory) value.append("d");
    else value.append("-");

    qsl.append(value);
    value = "";

    // and now for the file size.
    QString sz = "B";
    int tfsz = fileSize;
    if (tfsz > 1024) {
        tfsz = tfsz / 1024;
        sz = "KB";
    }
    if (tfsz > 1024) {
        tfsz = tfsz / 1024;
        sz = "MB";
    }
    if (tfsz > 1024) {
        tfsz = tfsz / 1024;
        sz = "GB";
    }
    value.append(QString::number(tfsz));
    value.append(sz);

    qsl.append(value);
    value = "";

    value.append("\t");
    value.append(QString::number(entryCluster));

    qsl.append(value);

    return qsl;
}
