#include "fat12_directory.h"

FAT12_Directory::FAT12_Directory()
{
    entries = new QList<FAT12_DirEntry *>();
    subdirs = new QList<FAT12_Directory *>();
}

FAT12_Directory::~FAT12_Directory() {
    // We need to delete all of our directory entries and subdirectories.
    qDebug() << "Performing destruction of Directory: " << directory_name;

    // First up - delete all of the entries.
    while (!entries->isEmpty())
        delete entries->takeFirst();
    delete entries;

    // Then delete all of the subdirectories
    while (!subdirs->isEmpty())
        delete subdirs->takeFirst();
    delete subdirs;
}

void FAT12_Directory::loadTree(QIODevice *iod, long start, int table_length,
                              FAT12_BPB *bpb, FAT12_FAT *fat) {

    // Clear the lists
    entries->clear();
    subdirs->clear();

    // Remember some important things
    this->BIOS_parameter_block = bpb;
    this->fat = fat;
    this->iod = iod;

    // Calculate how much data must be read
    int spcreq = bpb->max_root_entries * 32;

    // We will store some data in here
    QByteArray data;

    // Read the data in
    if (!iod->seek(start)) {
        // Something bad has happened
    }
    data = iod->read(spcreq);

    // Then continue loading the tree
    loadTree(data, iod, bpb, fat, "");
}
void FAT12_Directory::loadTree(QByteArray data, QIODevice *iod, FAT12_BPB *bpb,
                                FAT12_FAT *fat, QString name) {
    // Store a few important details
    directory_name = name;
    this->fat = fat;
    this->BIOS_parameter_block = bpb;
    this->iod = iod;

    // Figure out how many entries we have
    int entry_count = data.size() / 32; // Entries are 32 bytes long

    // Load the entries
    for (int i = 0; i < entry_count; i++) {
        // Read the entry data
        QByteArray entryData = data.mid(i*32,32);

        // See if we are at the end of the directory listing
        bool isEnd = true;
        for (int j = 0; j < 11; j++)
            if (entryData.at(j) != 0x00)
                isEnd = false;
        if (isEnd)
            break;


        // Create the entry:
        FAT12_DirEntry *ent = new FAT12_DirEntry();
        ent->load_entry(entryData);

        // We want to ignore those weird entries starting with the 0xE5 character.
        if (!ent->filename().startsWith(0xE5)) {

            // Store the entry for future use
            entries->append(ent);

            // If its a directory, create a new subdirectory, load it and
            // add it to the list. We want to ignore the "." and ".."
            // directories.
            if (ent->isDirectory() && ent->filename().trimmed() != "." &&
                ent->filename().trimmed() != "..") {

                FAT12_Directory *d = new FAT12_Directory();

                // Read in the directories data
                QByteArray directory_data = fat->loadFile(ent->getEntryCluster());


                // Load the subtree
                d->loadTree(directory_data, iod, bpb, fat,
                            ent->filename().trimmed());

                // And put it away for future use
                subdirs->append(d);
            }
        } else {
            // The filename started with 0xE5 - we currently ignore these so
            // they arent added to any of our lists. Therefore, we must delete
            // the entry right here, right now, so that we dont leak memory.
            delete ent;
        }
    }
}

QList<FAT12_DirEntry> FAT12_Directory::getEntries() {
    QList<FAT12_DirEntry> ent_list;

    int size = entries->size();

    for (int i = 0; i < size; i++) {
        FAT12_DirEntry *entry = entries->at(i);
        FAT12_DirEntry entry_copy = entry->copy();
        ent_list.append(entry_copy);
    }

    return ent_list;
}
FAT12_Directory * FAT12_Directory::getSubdirByName(QString name) {
    for (int i = 0; i < subdirs->size(); i++)
        if (subdirs->at(i)->getName().toUpper() == name.toUpper())
            return subdirs->at(i);

    return 0;
}
