#include "fat12.h"

FAT12::FAT12()
{
}

FAT12::~FAT12() {
    close();
    // iod is passed in and so should not be deleted.
}

void FAT12::load(QIODevice * device){
    // remember the IO Device!
    iod = device;

    // First, load the BIOS Parameter Block
    qDebug() << "Loading BIOS Parameter Block...";
    QByteArray sector;
    if (!device->seek(0)) {
        // Something has gone wrong.
    }
    sector = device->read(512);

    // Send away the BIOS Parameter Block data for processing
    bpb = new FAT12_BPB(sector);

    // Check for 0xAA55 starting at byte 510
    if (sector.at(510) == 0x55 && sector.at(511) == 0xAA)
        bootWord = true;
    else
        bootWord = false;

    // Now load the File Allocation Table.
    qDebug() << "Loading File Allocation Table...";
    first_fat = new FAT12_FAT(device, bpb);

    // Finally, load the filesystem tree
    qDebug() << "Loading Filesystem Tree...";

    // The table length is equal to size_of_entry * entries
    int tableLength = 32 * bpb->max_root_entries;

    // The root directory starts at track 0, head 1, sector 2. That is logical
    // sector 19 which is 9728 bytes into the device.

    // TODO: Allow the possibility of only one FAT or weird-sized FATs.
    long start = 9728;
    root_directory = new FAT12_Directory();
    root_directory->loadTree(device,start,tableLength,bpb,first_fat);

    qDebug() << "There are " << dir("/").size() << " entries in the root directory.";

    // Set the volume label.
    VolumeLabel = bpb->volume_label;

    QList<FAT12_DirEntry> ent = root_directory->getEntries();

    // A file with the VolumeID attribute set will override what ever
    // VolumeLabel is set in the BIOS Parameter Block.
    if (!ent.isEmpty()) {
        for (int i = 0; i < ent.size(); i++) {
            FAT12_DirEntry e = ent.at(i);
            // Long Filename entries have VolumeID, System, hidden and archive
            // attributes set. We must filter out these entries.
            if (e.isVolumeID() && !e.isSystemFile()) {
                VolumeLabel = e.filename();
                break;
            }
        }
    }

    // And we are done.
    qDebug() << "FAT12 Filesystem loaded and ready.";
    qDebug() << "Volume Label is " << VolumeLabel;
}

void FAT12::close(){
    // Delete everything we created.
    delete root_directory;
    delete first_fat;
    delete bpb;
    root_directory = 0;
    first_fat = 0;
    bpb = 0;

    // Set iod to 0 - a new one will be passed in when load is called next.
    iod = 0;

    // Reset bootWord to the default value.
    bootWord = false;
}

QList<FAT12_DirEntry> FAT12::dir(QString directory){

    // Get the desired directory:
    FAT12_Directory *dir = get_directory(directory);
    QList<FAT12_DirEntry> entries;

    // Could the directory be found?
    if (dir == 0) {
        // no, it couldnt. Return an empty entry list.
        return entries;
    }

    // Get the entries
    entries = dir->getEntries();

    // Remove any VolumeID entries
    for (int i = 0; i < entries.size(); i++){
        FAT12_DirEntry e = entries.at(i);
        if (e.isVolumeID())
            entries.removeAt(i);
    }

    // Now we just have to return a copy of its entry list.
    return entries;

}

QString FAT12::volumeLabel() const {
    return VolumeLabel;
}

bool FAT12::extractFile(QString source, QString dest) {
    // First, find the directory entry for the file.
    QStringList qsl = source.split("/",QString::SkipEmptyParts);
    QString filename = qsl.takeLast();

    QString directory;
    if (qsl.isEmpty()) directory = "/";
    else directory = qsl.join("/");

    if (root_directory == 0) QMessageBox::critical(0,"RootIsNull","RootIsNull");

    QList<FAT12_DirEntry> ents = dir(directory);

    for (int i = 0; i < ents.size(); i++) {
        FAT12_DirEntry e = ents.at(i);
        if (e.fullFileName().toUpper() == filename.toUpper()) {
            // This is the one we extract.
            return first_fat->extractFile(e,dest);
        }
    }

    QMessageBox::warning(0,"File Not Found", "The internal file " + source + " could not be found in the filesystem.");
    return false;
}

FAT12_Directory * FAT12::get_directory(QString directory) {

    // "/" is the root directory - if that is what we are after, return it.

    if (directory == "/")
        return root_directory;
    // Otherwise, continue searching.


    // First, lets get the path we must follow.
    QStringList path = directory.split('/');

    // Start at the root directory.
    FAT12_Directory *curdir = root_directory;
    while (!path.isEmpty()) {
        QString next_directory = path.takeFirst();

        // An empty directory name is probably just a typo - continue.
        if (next_directory.isEmpty()) continue;

        curdir = curdir->getSubdirByName(next_directory);
        if (curdir == 0) {
            /* getSubdirByName returned null - the subdirectory couldnt be
               found. That means that the directory we are searching for must
               not exist. Return null to indicate this.
             */
            qDebug() << "Directory " << directory << " not found.";
            return 0;
        }
    }
    return curdir;
}

void FAT12::printDebugInfo(){
    //
}

void FAT12::printDebugDirListing(QString directory){
    QString VID = bpb->volume_label;
    int fcount = 0, dcount = 0;
    QList<FAT12_DirEntry> entries = dir(directory);

    for (int i = 0; i < entries.size(); i++) {
        FAT12_DirEntry di = entries.at(i);
        if (di.isDirectory())
            dcount++;
        else
            fcount++;
    }


    // Prepare the Volume ID
    if (VID.isEmpty())
        qDebug() << " Volume in image has no label";
    else
        qDebug() << " Volume in image is " << VID;

    // Format the volume serial number properly
    QString VSN = QString::number(bpb->volume_id,16);

    qDebug() << " Volume Serial Number is " << VSN << "\n";
    qDebug() << " Directory of " << directory << "\n";

    for (int i = 0; i < entries.size(); i++) {
        FAT12_DirEntry di = entries.at(i);
        qDebug() << di.toString();
    }

    qDebug() << "\t\t" << QString::number(fcount,10) << " File(s)"; // bytes used
    qDebug() << "\t\t" << QString::number(dcount,10) << " Dir(s)"; // bytes free
}
