/* Project "DIMG" LibDsk Driver Plugin
 *
 * This Software (C) Copyright David Goodwin, 2008, 2009.
 *
 *   This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef _DQDLibDsk_H
#define _DQDLibDsk_H

// Qt Headers
#include <QString>
#include <QStringList>
#include <QByteArray>
#include <QFlags>
#include <QtPlugin>
#include <QFileDialog>
#include <QtDebug>
#include <QSettings>

// Plugin Headers
#include <DQDriver.h>
#include <DiskGeometry.h>

// LibDsk stuff
#include <libdsk.h>
#include "qlibdsk.h"

// DQDLibDsk headers
#include "constants.h"
#include "about.h"

#ifdef Q_OS_UNIX
#include "uxsettings.h"
#endif

/** LibDsk Drive Access Plugin.
 */
class DQDLibDsk : public QObject, public DQDskDriver {

    Q_OBJECT
    Q_INTERFACES(DQDskDriver)

public:

    DQDLibDsk() {}
    ~DQDLibDsk() {}

    DQDskDriver::Features getFeatures() const;

    int initialise();

    void displayAbout() const;

    void configure();

    QString getVersionString() const;

    QString getPluginName() const;

    int getLastErrorNumber() const;

    QString errToString(int errorCode) const;

    QStringList getDriveNames() const;

    QStringList getDescriptiveDriveNames() const;

    int openDrive(QString driveName);

    int closeDrive();

    QString currentDrive() const;

    bool isOpen() const;

    bool driveIsWritable(QString driveName) const;

    int writeData(int sector, const QByteArray *data);
    //int writeData(int cylinder, int head, int sector, const QByteArray *data);

    int readData(int sector, QByteArray *data);
    //int readData(int cylinder, int head, int sector, QByteArray *data);

    void exiting() { }

    DiskGeometry getDiskGeom();

    int setDiskGeom(QByteArray *boot_sector);
    int setDiskGeom(DiskGeometry dg);

private:
    /** Gets a list of removable media drives on the system.
      \returns A list of removable media drives on windows, or the
               floppy_drives as defined in constants.h on UNIX.
     */
    static QStringList getFloppyDrives();

    int lastError;  /*!< The last error that occured is remembered here. */

    bool driveOpen; /*!< If the drive is poen or not. */
    QString current_drive;  /*!< The currently open drive */

    QLibDsk drive;  /*!< The Qt LibDsk wrapper */
    DSK_GEOMETRY disk_geom; /*!< The geometry of the current disk */
    bool geom_ok; /*!< If set to False, get disk geometry before reading */

    /* Structures to support track-at-a-time reads
     */
    bool taatr_enable;      /*!< Enable Track-At-A-Time Reads */
    QByteArray taatr_buffer; /*!< Track-At-A-Time Read Buffer */
    dsk_pcyl_t taatr_cylinder; /*!< Track-At-A-Time Read cylinder number */
    dsk_phead_t taatr_head;     /*!< Track-At-A-Time Read head number */
    /** Reads a sector using Track-At-A-Time reads. If a track is not present in memory it will be read in.
      \param sector The sector to read
      \param data The data buffer to put the sector in
      \returns An error code
     */
    int taatr_readData(int sector, QByteArray *data);
    /** Refreshes the Track-At-A-Time track cache.
      \param geom The disk geometry
      \param cylinder The Track cylinder
      \param head The Track head
      \returns An error code
     */
    int taatr_read(const DSK_GEOMETRY *geom, dsk_pcyl_t cylinder, dsk_phead_t head);
};

#endif
