/* Project "DIMG" LibDsk Driver Plugin
 *
 * This Software (C) Copyright David Goodwin, 2008, 2009.
 *
 *   This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "DQDLibDsk.h"

// ------------------
// Win32 Stuff:
#ifdef Q_OS_WIN32
// This stuff is to support autodetecting removable media drives on Windows systems.
#include "windows.h"
static const QString win32_drives[] = {"A", "B", "C", "D", "E", "F",
                                       "G", "H", "I", "J", "K", "L",
                                       "M", "N", "O", "P", "Q", "R",
                                       "S", "T", "U", "V", "W", "X",
                                       "Y", "Z"};
static const int win32_drive_count = 26;
#endif


DQDskDriver::Features DQDLibDsk::getFeatures() const {
    DQDskDriver::Features f;
    f |= DQDskDriver::F_DisplayAbout;
    f |= DQDskDriver::F_GEOM_BOOTSEC;
#ifdef Q_OS_UNIX
    f |= F_Configure;
#endif
    return f;
}

int DQDLibDsk::initialise() {
    lastError = 0;
    driveOpen = false;
    geom_ok = false;

    QSettings settings;

    // Track-At-A-Time Reading stuff
    taatr_enable = settings.value("DQDLibDsk/taatr_enable","false").toBool();
    if (taatr_enable) qDebug() << "Track-At-A-Time Reads enabled";
    taatr_cylinder = 0;
    taatr_head = 0;

    return 0;
}

void DQDLibDsk::displayAbout() const  {
    About a;
    a.exec();
}

void DQDLibDsk::configure()  {
#ifdef Q_OS_UNIX
    uxsettings uxs;
    // Load the settings
    QSettings settings;

    QStringList fdl = settings.value("DQDLibDsk/drives", QString("/dev/fd0,/dev/fd1").split(",")).toStringList();

    uxs.add_floppy_drives(fdl);

    int result = uxs.exec();
    if (result == QDialog::Accepted) {
        settings.setValue("DQDLibDsk/drives",uxs.get_floppy_drives());
    }
#endif
}

QString DQDLibDsk::getVersionString() const {
    return "1.0.0";
}

QString DQDLibDsk::getPluginName() const {
    return "LibDsk Drive Access Plugin";
}

int DQDLibDsk::getLastErrorNumber() const {
    return lastError;
}

QString DQDLibDsk::errToString(int errorCode) const {
    // LibDsk error codes start at 0 and go down (-1, -2, etc).
    // Our own error codes (and those of QtDIMG) start at 0 and go up.
    // Therefore, if it is bigger than 0 we must handle it, if it is
    // smaller than 0 it is LibDsks error to interpret.
    if (errorCode > 0) {
        // It is a QtDIMG or DQDLibDsk error code
        switch (errorCode) {
            case ERR_NOTIMPL:
                return "Function Not Implemented.";
                break;
            case ERR_USRCANCVIRTMNT:
                return "User canceled opening of virtual floppy drive.";
                break;
            default:
                return "Unknown Error.";
                break;
        }
    } else if (errorCode == 0) {
        // There was no error
        return "No Error.";
    } else {
        // It is a LibDsk error. Let it figure out what it means.
        return QLibDsk::strerror(errorCode);
    }

    return "Invalid Error Code.";
}

QStringList DQDLibDsk::getDriveNames() const {
    QStringList fd = getFloppyDrives();

    if (enable_virtual_drive)
        fd << "::VIRTDRV";

    return fd;
}

QStringList DQDLibDsk::getDescriptiveDriveNames() const {
    QStringList descName;
    QStringList dn = getDriveNames();

    for (int i = 0; i < descName.size(); i++) {
        dn.append("Drive " + descName.at(i));
    }

    if (enable_virtual_drive) {
        dn.removeLast();
        dn << "Virtual Drive";
    }

    return dn;
}

QStringList DQDLibDsk::getFloppyDrives() {
        QStringList qsl;
        for (int i = 0; i < floppy_drive_count; i++)
                qsl.append(QString(floppy_drives[i]));


        if (detect_floppy_drives) {
#ifdef Q_OS_WIN32
                qsl.clear();
                for (int j = 0; j < win32_drive_count; j++) {
                        UINT dt = GetDriveTypeA(QString(win32_drives[j] + ":") \
                                        .toAscii().constData());

                        QString msg = "Drive ";
                        msg.append(win32_drives[j]);
                        msg.append(" is ");

                        if (dt == DRIVE_UNKNOWN)
                                msg.append("DRIVE_UNKNOWN");
                        else if (dt == DRIVE_NO_ROOT_DIR)
                                msg.append("DRIVE_NO_ROOT_DIR");
                        else if (dt == DRIVE_REMOVABLE) {
                                msg.append("DRIVE_REMOVABLE");
                                qsl.append(QString(win32_drives[j]) + ":");
                        }
                        else if (dt == DRIVE_FIXED)
                                msg.append("DRIVE_FIXED");
                        else if (dt == DRIVE_REMOTE)
                                msg.append("DRIVE_REMOTE");
                        else if (dt == DRIVE_CDROM)
                                msg.append("DRIVE_CDROM");
                        else if (dt == DRIVE_RAMDISK)
                                msg.append("DRIVE_RAMDISK");
                        else
                                msg.append("unknown :(");
                }
#endif
#ifdef Q_OS_UNIX
            QSettings settings;
            return settings.value("DQDLibDsk/drives", QString("/dev/fd0,/dev/fd1").split(",")).toStringList();
#endif
        }

        return qsl;
}

int DQDLibDsk::openDrive(QString driveName) {

    // If the virtual drive has been selected, get a filename
    // from the user. Does not support creating new images.
    if (enable_virtual_drive && driveName == "::VIRTDRV") {
        QString fn = QFileDialog::getOpenFileName(0,"Open Disk Image for Virtual Floppy Drive");
        if (fn.isEmpty()) return ERR_USRCANCVIRTMNT;
        driveName = fn;
    }

    dsk_err_t err;

    QSettings settings;
    QString driver_to_use = settings.value("DQDLibDsk/driver","floppy").toString();

    // Figure out which driver we should use.
    if (driver_to_use == "ntwdm") {
        qDebug() << "--> Opening drive using NTWDM Driver";
        err = drive.open(driveName,"ntwdm",0);
    } else {
        qDebug() << "--> Opening drive using floppy driver";
        err = drive.open(driveName,0,0);
    }

    if (err) {
        // An error has occured opening the device.
        lastError = err;
        return lastError;
    }

    // 5 seems like a good number of times to retry.
    err = drive.set_retry(5);
    if (err) {
        // An error has occured setting the retry count.
        lastError = err;
        return lastError;
    }

    // The opening of the drive was successfull. Remember this.
    driveOpen = true;
    current_drive = driveName;

    lastError = err;
    return lastError;
}

int DQDLibDsk::closeDrive() {
    /* Here we need to:
         - Close the current drive
     */

    lastError = drive.close();

    if (!lastError) {
        driveOpen = false;
        current_drive = "";
    }

    geom_ok = false;

    return lastError;
}

QString DQDLibDsk::currentDrive() const {
    /* Here we need to:
        - Return the name of the currently open drive or "" if there is no open drive.
     */
    return current_drive;
}

bool DQDLibDsk::isOpen() const {
    /* Here we need to:
        - Return true if the drive is currently open, otherwise false.
     */
    return driveOpen;
}

bool DQDLibDsk::driveIsWritable(QString driveName) const {
    /* We will just assume all floppy disks are writable. If its write
       protected an error will occur when a write is attempted.
     */
    return true;
}

int DQDLibDsk::writeData(int sector, const QByteArray *data) {

    qDebug() << "Logical Sector: " << sector;
    lastError = drive.lwrite(&disk_geom,data->data(),sector);

    // And we are now done
    return lastError;
}

int DQDLibDsk::readData(int sector, QByteArray *data) {
    /* Here we need to:
          1. Figure out the Cylinder, Head and Sector from the logical sector
          2. Read that CHS sector
          3. Put the results into data
          4. Return lastError.
     */

    // If we are supposed to be doing Track-At-A-Time reads then do that.
    if (taatr_enable)
        return taatr_readData(sector,data);

    // Otherwise, read the sector

    // Disk Geometry is required for this exercise
    if (!geom_ok) {
        qDebug() << "Geometry not set. Asking LibDsk to guess...";
        lastError = drive.getgeom(&disk_geom);
        if (lastError) return lastError;
        geom_ok = true;
    }

    // Allocate a buffer
    char *buf = NULL;
    if (buf == NULL) {
        buf = (char *)dsk_malloc(disk_geom.dg_secsize);
    }

    if (!buf) lastError = DSK_ERR_NOMEM;
    if (lastError) {
        if (buf) dsk_free(buf);
        return lastError;
    }

    // Read the sector into the buffer.
    lastError = drive.lread(&disk_geom,buf,sector);
    if (lastError) {
        if (buf) dsk_free(buf);
        return lastError;
    }

    // Copy data from the buffer to the QByteArray.
    data->append(QByteArray(buf,disk_geom.dg_secsize));

    if (buf) dsk_free(buf);

    return lastError;
}

int DQDLibDsk::taatr_readData(int sector, QByteArray *data) {
    // Disk Geometry is required for this exercise
    if (!geom_ok) {
        qDebug() << "Geometry not set. Asking LibDsk to guess...";
        lastError = drive.getgeom(&disk_geom);
        if (lastError) return lastError;
        geom_ok = true;
    }

    // We need to throw this into CHS form
    dsk_pcyl_t mtaatr_cyl = 0;
    dsk_phead_t mtaatr_hd = 0;
    dsk_psect_t mtaatr_sec = 0;

    lastError = dg_ls2ps(&disk_geom,
              /* in */   sector,
              /* out */  &mtaatr_cyl, &mtaatr_hd, &mtaatr_sec);

    if (lastError)
        return lastError;

    // If the cylinder or head is different from what we currently have
    // cached, read in the new track
    if (taatr_cylinder != mtaatr_cyl || taatr_head != mtaatr_hd || taatr_buffer.isEmpty()) {
        if (taatr_cylinder != mtaatr_cyl) qDebug() << "Cylinder changed - refreshing track buffer.";
        if (taatr_head != mtaatr_hd) qDebug() << "Head changed - refreshing track buffer.";
        if (taatr_buffer.isEmpty()) qDebug() << "track buffer empty - refreshing.";

        lastError = taatr_read(&disk_geom, mtaatr_cyl,mtaatr_hd);
        if (lastError) {
            qDebug() << "Error refreshing track buffer";
            //if (taatr_buffer) dsk_free(taatr_buffer);
            return lastError;
        }
    }

    // Ok, we should have the track in memory. Time to pull out the
    // requested sector.
    qDebug() << "Reading sector from buffer at position " << ((mtaatr_sec-1) * disk_geom.dg_secsize);
    data->append(taatr_buffer.mid((mtaatr_sec-1) * disk_geom.dg_secsize, disk_geom.dg_secsize));

    // And we are done.
    return lastError;
}

int DQDLibDsk::taatr_read(const DSK_GEOMETRY *geom, dsk_pcyl_t cylinder, dsk_phead_t head) {

    qDebug() << "Reading track at cyl: " << cylinder << " head: " << head;

    char * mtbuf = 0;

    // Deallocate existing buffer
    taatr_buffer.clear();

    // Allocate new buffer
    qDebug() << "->Allocating buffer";
    mtbuf = (char *)dsk_malloc(disk_geom.dg_sectors*disk_geom.dg_secsize);

    qDebug() << "->Reading track...";
    taatr_head = head;
    taatr_cylinder = cylinder;
    lastError = drive.ptread(geom,mtbuf, cylinder, head);

    // Store track
    taatr_buffer.append(QByteArray(mtbuf,disk_geom.dg_sectors*disk_geom.dg_secsize));

    return lastError;
}

DiskGeometry DQDLibDsk::getDiskGeom() {
    // We will store some geometry info in this
    DiskGeometry dg;

    // We will get that info from this
    DSK_GEOMETRY lddg;

    // Grab the geometry info from cache if its there
    if (!geom_ok) {
        qDebug() << "getDiskGeom(): Requesting Disk Geometry from LibDsk...";
        lastError = drive.getgeom(&lddg);
    } else {
        qDebug() << "getDiskGeom(): Using cached Disk Geometry...";
        lddg = disk_geom;
    }

    // If an error occured just return empty geometry info
    if (lastError) {
        qDebug() << "getDiskGeom(): An error occured. Returning empty diskgeom.";
        return dg;
    }

    // Now pull out the geometry info
    qDebug() << "getDiskGeom(): Translating Disk Geometry Information...";
    dg.cylinders = lddg.dg_cylinders;
    dg.heads = lddg.dg_heads;
    dg.sectors = lddg.dg_sectors;
    dg.sectorSize = lddg.dg_secsize;

    qDebug() << "getDiskGeom(): Complete";
    // And return it.
    return dg;
}

int DQDLibDsk::setDiskGeom(QByteArray *boot_sector){
    /* Normally LibDsk would give us the disk geometry. Problem is, LibDsk can
       not see the disk image as the host program is managing it. So we can not
       just ask LibDsk for the geometry. We have to try a few different
       geometry types until we get one that works.

       Function     Boot Sector Size
       dg_dosgeom   512 bytes
       dg_pcwgeom   512 bytes by the looks of it
       dg_cpm86geom 512 bytes
       dg_aprigeom  assuming 512bytes
     */

    int e;

    qDebug() << "Attempting to guess disk geometry based on boot sector...";
    // Assume sector size is 512 bytes - it seems to be enough for all of them.
    unsigned char* buf = NULL;
    if (buf == NULL) {
        buf = (unsigned char*)dsk_malloc(512);
    }
    if (!buf) lastError = DSK_ERR_NOMEM;
    if (lastError) {
        if (buf) dsk_free(buf);
        return lastError;
    }

    // Now copy over the sector into the buffer
    for (int i = 0; i < 512; i++)
        buf[i] = boot_sector->at(i);

    // Try to guess the disk geometry
    qDebug() << "Attempting DOS Geometry...";
    e = dg_dosgeom(&disk_geom, buf);
    if (e == DSK_ERR_BADFMT) {
        qDebug() << "Not DOS Geometry. Attempting PCW Geometry...";
        e = dg_pcwgeom(&disk_geom, buf);
    }
    if (e == DSK_ERR_BADFMT) {
        qDebug() << "Not PCW Geometry. Attempting Apricot Geometry...";
        e = dg_aprigeom(&disk_geom, buf);
    }
    if (e == DSK_ERR_BADFMT) {
        qDebug() << "Not Apricot Geometry. Attempting CPM86 Geometry...";
        e = dg_cpm86geom(&disk_geom, buf);
    }
    if (e == DSK_ERR_BADFMT) {
        qDebug() << "Not CPM86 Geometry.";
        qDebug() << "WARNING: Unable to guess disk geometry.";
    }

    // We will have either had an error or the Disk Geometry should be in
    // disk_geom.

    // LibDsk seems to be unable to figure out the sidedness on its own.
    // Doing this means we loose support for CP/M-86 and Acorn formatted
    // disks but that isnt a huge issue - I don't have any disks of either
    // sort anyway.
    disk_geom.dg_sidedness = SIDES_ALT; // For IBM-PC disks.

    if (buf) dsk_free(buf);

    if (!e) geom_ok = true;

    return e;
}

int DQDLibDsk::setDiskGeom(DiskGeometry dg){
    // Does nothing - we need more detail than the DiskGeometry data structure holds.
    return 1;
}

Q_EXPORT_PLUGIN2(dqlddrv, DQDLibDsk)
