#include "imager.h"

Imager::Imager(QObject *parent): QObject(parent)
{
    dqdrv = 0;
    pw = 0;
}

Imager::~Imager() {
    if (dqdrv) {
        dqdrv->exiting();
        delete dqdrv;
        dqdrv = 0;
    }
}

int Imager::read_image(QIODevice * destination_device) {
    // Some useful variables
    int errcode = 0;
    QByteArray buf;

    // Open the drive
    errcode = open_current_drive();
    if (errcode) return errcode;

    // Drive should be open. Now get disk geometry.
    DiskGeometry dg = dqdrv->getDiskGeom();

    if (dqdrv->getLastErrorNumber() != errors::E_SUCCESS) {
        qDebug() << "Failed to get disk geometry. Error " << QString::number(dqdrv->getLastErrorNumber(),10);
        // Something bad happened
        QMessageBox::critical(0,"Error Getting Disk Geometry",dqdrv->errToString(dqdrv->getLastErrorNumber()));
        dqdrv->closeDrive();
        return errors::E_USER_CANCELED;
    }
    qDebug() << "Got Disk Geometry.";

    // Now to open the image
    qDebug() << "Opening destination image...";
    if (!destination_device->open(QIODevice::ReadWrite)) {
        qDebug() << "Unable to open destination image. Aborting.";
        return errors::E_CANT_OPEN_DEST_DEV;
    }
    errcode = 0;

    // Prepare the progress dialog
    qDebug() << "Setup progress dialog...";
    int max = dg.heads * dg.cylinders * dg.sectors;
    QString prog_format_string = "Reading from " + currentDrive + " to image";
    QProgressDialog *progress = 0;
    if (pw)
        progress = new QProgressDialog("Reading Disk","Cancel",0,max,pw);
    else
        progress = new QProgressDialog("Reading Disk","Cancel",0,max,0);
    progress->setWindowTitle("Reading Disk");
    progress->setLabelText(prog_format_string);
    progress->setWindowModality(Qt::WindowModal);
    QString qsCyl, qsCylT, qsHead, qsHeads, qsEct, qsEcT, qsLSec, qsLSMax;
    qsLSMax.setNum(max,10);

    qDebug() << "Imaging Disk...";
    int cyl, head, sec; // The CHS parameters
    int logicalsec = -1;
    for (cyl = 0; cyl < dg.cylinders; ++cyl) {
        for (head = 0; head < dg.heads; ++head) {
            for (sec = 0; sec < dg.sectors; ++sec) {
                // Update the logical sector
                logicalsec++;

                // Update progress information
                qsCyl.setNum(cyl + 1);
                qsCylT.setNum(dg.cylinders);
                qsHead.setNum(head + 1);
                qsHeads.setNum(dg.heads);
                qsEct.setNum(sec);
                qsEcT.setNum(dg.sectors - 1);
                qsLSec.setNum(logicalsec);
                progress->setLabelText("Reading cylinder " + qsCyl + "/" + qsCylT +
                                      " Head " + qsHead + "/" + qsHeads + " Sector " +
                                      qsEct + "/" + qsEcT);
                progress->setValue(dg.sectors * (cyl * dg.heads + head) + sec);
                qDebug() << "Attempting Logical Sector " +
                        qsLSec + "/" + qsLSMax +
                        " (Cyl: " + qsCyl + " Head: " +
                        qsHead + " Sec: " +
                        qsEct + ")";

                // If the user hit cancel, stop.
                if (progress->wasCanceled()) {
                    qDebug() << "->User aborted";
                    //delete buf;
                    destination_device->close();
                    dqdrv->closeDrive();
                    delete progress;
                    dqdrv->closeDrive();
                    return errors::E_USER_CANCELED;
                }

                // Now try to actually read the sector. Keep looping until we
                // are successful or the user gives up
                while (true) {
                    // Clear the buffer and read the sector
                    buf.clear();
                    errcode = dqdrv->readData(logicalsec,&buf);

                    if (buf.size() < dg.sectorSize) {
                        qDebug() << "WARNING: Data read does not match sector size. Retrying";
                    }

                    // If there was an error, ask the user what to do
                    if ((errcode != errors::E_SUCCESS)) {
                        qDebug() << "Error reading sector. Prompting user.";
                        // There was an error
                        QMessageBox msgb;

                        msgb.setText(dqdrv->errToString(errcode));
                        msgb.setStandardButtons(QMessageBox::Abort | QMessageBox::Retry | QMessageBox::Ignore);
                        msgb.setDefaultButton(QMessageBox::Abort);
                        msgb.setIcon(QMessageBox::Critical);
                        int ret = msgb.exec();

                        switch (ret) {
                            case (QMessageBox::Abort):
                                // Something bad has happened and we are giving up.
                                // Tidy up, close everything we've opened, etc.
                                qDebug() << "->User chose to abort.";
                                //delete buf;
                                destination_device->close();
                                dqdrv->closeDrive();
                                delete progress;
                                dqdrv->closeDrive();
                                return errors::E_USER_CANCELED;
                                break;
                            case (QMessageBox::Retry):
                                qDebug() << "->User chose to retry.";
                                // We will attempt to read again.
                                continue;
                            case (QMessageBox::Ignore):
                                // We will write out an empty sector in the place of this one.
                                qDebug() << "->User chose to ignore.";
                                buf.resize(dg.sectorSize);
                                for (int i = 0; i < dg.sectorSize; i++)
                                    buf[i] = 0x00;
                                errcode = errors::E_SUCCESS;
                                break;
                            default:
                                // This should never happen
                                break;
                        }
                    }

                    if (errcode == errors::E_SUCCESS) {
                        qDebug() << "Appending sector to image";
                        destination_device->seek(logicalsec * dg.sectorSize);
                        destination_device->write(buf);
                        //out << buf;
                        break;
                    }
                }
            }
        }
    }

    // Close the device, etc.
    destination_device->close();
    dqdrv->closeDrive();
    delete progress;
    return errors::E_SUCCESS;
}

int Imager::write_image(QIODevice * source_device) {
    // Some useful variables
    int errcode = 0;
    QByteArray buf;

    // Open the drive
    errcode = open_current_drive();
    if (errcode) return errcode;

    // Now open the source device
    qDebug() << "Opening source image...";
    if (!source_device->open(QIODevice::ReadWrite)) {
        qDebug() << "Unable to open source image. Aborting.";
        return errors::E_CANT_OPEN_SRC_DEV;
    }
    errcode = 0;

    // The drive should now be open. Next we have to set the disk geometry
    // to write with. We need the boot sector for this. Grab the first 1KB.
    qDebug() << "Attempting to set Geometry...";
    buf = source_device->read(1024);
    errcode = dqdrv->setDiskGeom(&buf);
    if (errcode != errors::E_SUCCESS) {
        qDebug() << "Failed to guess disk geometry. Error " << QString::number(dqdrv->getLastErrorNumber(),10);
        // Something bad happened
        QMessageBox::critical(0,"Error Setting Disk Geometry",dqdrv->errToString(dqdrv->getLastErrorNumber()));
        dqdrv->closeDrive();
        return errcode;
    }
    DiskGeometry dg = dqdrv->getDiskGeom();
    qDebug() << "Cylinders: " << dg.cylinders;
    qDebug() << "Heads: " << dg.heads;
    qDebug() << "Sectors: " << dg.sectors;
    qDebug() << "Sector Size: " << dg.sectorSize;
    qDebug() << "Geometry Set.";

    // Prepare the progress dialog
    qDebug() << "Setup progress dialog...";
    int max = dg.heads * dg.cylinders * dg.sectors;
    QString prog_format_string = "Reading from " + currentDrive + " to image";
    QProgressDialog *progress = 0;
    if (pw)
        progress = new QProgressDialog("Writing Disk","Cancel",0,max,pw);
    else
        progress = new QProgressDialog("Writing Disk","Cancel",0,max,0);
    progress->setWindowTitle("Writing Disk");
    progress->setLabelText(prog_format_string);
    progress->setWindowModality(Qt::WindowModal);
    QString qsCyl, qsCylT, qsHead, qsHeads, qsEct, qsEcT, qsLSec, qsLSMax;
    qsLSMax.setNum(max,10);

    // Now image the disk
    qDebug() << "Imaging Disk...";
    int cyl, head, sec; // The CHS parameters
    int logicalsector = -1;
    for (cyl = 0; cyl < dg.cylinders; ++cyl) {
        for (head = 0; head < dg.heads; ++head) {
            for (sec = 0; sec < dg.sectors; ++sec) {
                // Update the logical sector
                logicalsector++;

                // Update progress information
                qsCyl.setNum(cyl + 1);
                qsCylT.setNum(dg.cylinders);
                qsHead.setNum(head + 1);
                qsHeads.setNum(dg.heads);
                qsEct.setNum(sec);
                qsEcT.setNum(dg.sectors - 1);
                qsLSec.setNum(logicalsector);
                progress->setLabelText("Writing cylinder " + qsCyl + "/" + qsCylT +
                                      " Head " + qsHead + "/" + qsHeads + " Sector " +
                                      qsEct + "/" + qsEcT);
                progress->setValue(dg.sectors * (cyl * dg.heads + head) + sec);
                qDebug() << "Attempting Logical Sector " +
                        qsLSec + "/" + qsLSMax +
                        " (Cyl: " + qsCyl + " Head: " +
                        qsHead + " Sec: " +
                        qsEct + ")";

                // If the user hit cancel, stop.
                if (progress->wasCanceled()) {
                    qDebug() << "->User aborted";
                    source_device->close();
                    dqdrv->closeDrive();
                    delete progress;
                    dqdrv->closeDrive();
                    return errors::E_USER_CANCELED;
                }

                // Fetch the logical sector from the file
                source_device->seek(logicalsector * dg.sectorSize);
                buf.clear();
                buf = source_device->read(dg.sectorSize);
                qDebug() << "Read bytes from image: " << buf.size();

                // Now try to actually write the sector. Keep looping until we
                // are successful or the user gives up
                while (true) {
                    // Write the sector
                    errcode = dqdrv->writeData(logicalsector,&buf);
                    //errcode = dqdrv->writeData(cyl,head,sec+1,&buf);

                    // If there was an error, ask the user what to do
                    if ((errcode != errors::E_SUCCESS)) {
                        qDebug() << "Error writing sector. Prompting user.";
                        // There was an error
                        QMessageBox msgb;

                        msgb.setText(dqdrv->errToString(errcode));
                        msgb.setStandardButtons(QMessageBox::Abort | QMessageBox::Retry | QMessageBox::Ignore);
                        msgb.setDefaultButton(QMessageBox::Abort);
                        msgb.setIcon(QMessageBox::Critical);
                        int ret = msgb.exec();

                        switch (ret) {
                            case (QMessageBox::Abort):
                                // Something bad has happened and we are giving up.
                                // Tidy up, close everything we've opened, etc.
                                qDebug() << "->User chose to abort.";
                                //delete buf;
                                source_device->close();
                                dqdrv->closeDrive();
                                delete progress;
                                dqdrv->closeDrive();
                                return errors::E_USER_CANCELED;
                                break;
                            case (QMessageBox::Retry):
                                qDebug() << "->User chose to retry.";
                                // We will attempt to read again.
                                continue;
                            case (QMessageBox::Ignore):
                                // We will write out an empty sector in the place of this one.
                                qDebug() << "->User chose to ignore.";
                                errcode = errors::E_SUCCESS;
                                break;
                            default:
                                // This should never happen
                                break;
                        }
                    }

                    if (errcode == errors::E_SUCCESS) {
                        qDebug() << "Sector Written.";
                        break;
                    }
                }
            }
        }
    }

    // Close the device, etc.
    source_device->close();
    dqdrv->closeDrive();
    delete progress;
    return errors::E_SUCCESS;
}

int Imager::open_current_drive() {
    int errcode = 0;
    qDebug() << "Attempting to open drive " << get_current_drive() << "...";
    while (!dqdrv->isOpen()) {
        errcode = dqdrv->openDrive(get_current_drive());
        qDebug() << "OpenDrive() = " << errcode;
        if (errcode != errors::E_SUCCESS) {
            // We ran into problems opening the drive. Get a message for the user
            QMessageBox msgb;
            msgb.setText("Error opening drive: " + dqdrv->errToString(errcode));
            msgb.setStandardButtons(QMessageBox::Abort | QMessageBox::Retry);
            msgb.setDefaultButton(QMessageBox::Abort);
            msgb.setIcon(QMessageBox::Warning);
            int ret = msgb.exec();

            // Abort if the user says so
            if (ret == QMessageBox::Abort) {
                qDebug() << "Aborting...";
                return errors::E_USER_CANCELED;
            }
            // Otherwise try again
        }
    }
    qDebug() << "Drive Open.";
    return errors::E_SUCCESS;
}

int Imager::load_drivers() {
    if (loadDQPlugin())
        return errors::E_SUCCESS;
    else
        return errors::E_FAILURE;
}

bool Imager::loadDQPlugin() {
    QString plugin_file, alt_file, specified_file;

    // Set the directory for the plugin to be the applications directory
    specified_file = plugin_file = alt_file = QApplication::applicationDirPath();

    QSettings settings;
    QString tmp = settings.value("Plugins/DQDriver","").toString();

    // Build the base filename
#ifdef Q_OS_WIN32
    plugin_file += "/DQWindrv";
    alt_file += "/dqplugin";
    specified_file += "/" + tmp;
#else
    plugin_file += "/libDQlibdskdrv";
    alt_file += "/libdqplugin";
    specified_file += "/lib" + tmp;
#endif

    // If it is a debug build, use the debug build of the plugin
#ifdef _DIMG_DEBUG
    plugin_file += "d";
    alt_file += "d";
    specified_file += "d";
#endif

    // Add on the extension
#ifdef Q_OS_WIN32
    plugin_file += ".dll";
    alt_file += ".dll";
    specified_file += ".dll";
#else
    plugin_file += ".so";
    alt_file += ".so";
    specified_file += ".so";
#endif

    QFileInfo info(specified_file);

    if (info.exists())
        plugin_file = specified_file;
    else {
        info.setFile(plugin_file);
        if (!info.exists()) {
            plugin_file = alt_file;
            info.setFile(plugin_file);
            if (!info.exists()) {
                plugin_file = QFileDialog::getOpenFileName(0,"Locate Disk Driver Plugin");
                if (plugin_file.isEmpty()) {
                    QMessageBox::critical(0,"Plugin Load Error","No Disk Driver plugin specified. Exiting.");
                }
            }
        }
    }

    qDebug() << "Creating plugin loader for file " << plugin_file;
    QPluginLoader *loader = new QPluginLoader(plugin_file);
    qDebug() << "Loading plugin...";
    if (loader->load()) {
        dqdrv = qobject_cast<DQDskDriver *>(loader->instance());
        qDebug() << "ok";
        if (!dqdrv) {
            QMessageBox::critical(0,"Plugin Load Error","Loaded plugin was not of type DQDskDriver.");
        } else {
            dqdrv->initialise();
        }
    }
    else {
        QMessageBox::critical(0,"Plugin Load Error","The DQDskDriver plugin could be found but not loaded.\nThis is likely a problem with the plugin.\nThe Plugin File is:\n" + plugin_file);
    }
    delete loader;
    loader = 0;
    return (dqdrv);
}

void Imager::set_current_drive(QString driveName) {
    currentDrive = driveName;
}

QString Imager::get_current_drive() {
    return currentDrive;
}

QStringList Imager::getDriveNames() {
    if (dqdrv)
        return dqdrv->getDriveNames();
    return QStringList();
}

QStringList Imager::getDescriptiveDriveNames() {
    if (dqdrv)
        return dqdrv->getDescriptiveDriveNames();
    return QStringList();
}

bool Imager::driver_ready() {
    return dqdrv;
}

QString Imager::get_driver_name() {
    if (!dqdrv) return "";
    return dqdrv->getPluginName();
}
QString Imager::get_driver_version() {
    if (!dqdrv) return "";
    return dqdrv->getVersionString();
}

bool Imager::configurable() {
    if (dqdrv != NULL) {
        if (dqdrv->getFeatures().testFlag(dqdrv->F_Configure)) {
            qDebug() << "DQDskDriver is Configurable";
            return true;
        } else {
            qDebug() << "DQDskDriver is NOT Configurable";
            return false;
        }
    }
    else return false;
}

void Imager::configure() {
    if (dqdrv != NULL)
        dqdrv->configure();
}
