/*
-----------------------------------------------------
   Application Name: "DIMG"
Application Version: 1.0.0
          File Name: Dimg.h
         Start Date: 27/06/2008
          Author(s): David Goodwin
          Copyright: (C) Copyright David Goodwin, 2008, 2009
            License: GNU General Public License
   File Description: DIMG Class header
-----------------------------------------------------
	This file is part of "DIMG".

    This is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This software is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this software; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

	File Notes:
	------------------
*/


#ifndef DIMG_H
#define DIMG_H

// ------------------
// Headder files:
// ------------------
// Qt
#include <QObject>
#include <QMainWindow>
#include <QApplication>
#include <QString>
#include <QStringList>
#include <QProgressDialog>
#include <QMessageBox>
#include <QFileDialog>
#include <QPointer>
#include <QtDebug>
#include <QPluginLoader>
#include <QFileInfo>
#include <QBuffer>
#include <QCloseEvent>

// DIMG Plugin Headers
#include <FSPlugin.h>
#include <DQDriver.h>
#include <fsentry.h>

// DIMG headers
#include "ui_main_window.h"
#include "fileproperties.h"
#include "constants.h"
#include "errors.h"
#include "errorstrings.h"
#include "utils.h"
#include "aboutdialog.h"
#include "imager.h"


/**
\brief The Dimg class manages the MainWindow UI and other program functionality.

This class essentially glues the various components of DIMG together.
*/
class Dimg : public QMainWindow
{
      Q_OBJECT

public:
	// Public Functions:
	Dimg(QApplication *app);	/*!< The Class Constructor */
        ~Dimg();    /*!< Class destructor */

        /** Loads the specified image file.

          \param filename The image filename
         */
        void loadFile(QString filename);

        /** Extracts the specified file from the currently loaded image.
          \param filename The file to extract.
         */
        void extractFile(QString filename);
        /** Shows the properties dialog for the specified file in the currently
          loaded image.
          \param filename The file to show the properties dialog for.
         */
        void showFileProperties(QString filename);

        /** Returns the file currently selected in the Tree Widget.
          If no file is selected, an empty string is returned.
          \returns The currently selected file name or an empty string.
         */
        QString getSelectedFile();


private slots:
        /** Selects the disk drive represented by the supplied QAction.

          \param dskAction The action that represents the disk drive
         */
	void diskSelect(QAction * dskAction);

        /** Displays a File Open Dialog and then calls loadFile(QString).
         */
        void loadFile();


        /** Extracts the file currently selected in the UIs Tree Widget.
         */
        void extractFile();

        /** Shows the properties for the currently selected file in the UIs Tree Widget.
         */
        void showFileProperties();

        /** The close event - asks the user to save, tidies up, etc.
          */
        void closeEvent(QCloseEvent *event);

        /** Closes the currently open disk image without saving.
         */
        void closeImage();

        /** Creates an empty disk image in memory. It does not setup
          the filesystem or its view.
          */
        void createImage();

        /** Loads the filesystem view for the currently open image.
          */
        void loadFileSystem();

        /** Shows the about dialog
         */
        void showAbout();

        /** Images the media in the currently selected drive.
         */
        void imageCurrentDrive();

        /** Writes the current image to the current drive.
          */
        void writeCurrentDrive();

        /** Saves the currently loaded image.

          \returns An error code
         */
        int saveCurrentImage();

        /** Displays a file save dialog. If the user accepts it sets the filename
            and then calls saveCurrentImage().

          \returns The error code from saveCurrentImage or  errors::E_USER_CANCELED
                if the user canceled.
         */
        int saveCurrentImageAs();

        /** Resizes the columns when ever something is expanded or collapsed.
          */
        void resize_columns(const QModelIndex &index);

private:
        // Private methods

        /** Loads the filesystem plugin.
          \returns True if the plugin was loaded successfully.
         */
        bool loadFSPlugin();

        /** Prepares the filesystem tree widget for use. Adds column headers, etc.
         */
        void prepareFSTreeView();

        /** Sets the programs window title to include the filename of the currently open
          image file (if any).
         */
        void setProgramWindowTitle();

        /** Resets the main window title to include the specified filename.
            If the imageDirty flag is set then an asterisk will be appended
            to the end of the displayed filename.

            By default the format is "{Application-name} version {version-string} - [{filename}]. If no
            filename is supplied, the format is "{Application-name} version {version-string}".

            \param filename The name of the file to show in the title bar.
         */
        void setProgramWindowTitle(QString filename);

        /** Loads the filesystem tree from the currently open FSPlugin.
          \param parent The parent QWidgetItem. If this is 0, items will be
          inserted into the root.
          \param directory The directory to start loading. If loading the root
          directory, "/" should be supplied here along with 0 as the parent
          parameter.
         */
        void loadFSTree(QTreeWidgetItem *parent, QString directory);

        /** Asks the user if the currently open image should be saved if an
          image is open and is dirty (unsaved).
         */
        int prompt_image_save();

	// Private variables
	Ui::MainWindowUI ui; /*!< The main window ui object */
	QApplication *pApp;	 /*!< The application object */

	// We need to keep track of disk select actions so we chan handle checking
	// properly
        QList<QAction *> diskSelActions;    /*!< A list of disk selection actions */

        // The disk imager
        Imager *imgr;   /*!< The disk imager */

        // The filesystem and disk image
        FSPlugin *fs;    /*!< The current File System Plugin */
        //DQDskDriver *dqdrv;
        QFile *file;     /*!< The currently loaded image file */
        QBuffer *img;    /*!< The disk image in memory */
        QString filename;/*!< The filename for the image currently in memory */
        bool imageDirty; /*!< Unsaved changes have been made to the image in-memory */
        bool imgOpen;    /*!< If an image is open or not */
};

#endif // DIMG_H
