/*
-----------------------------------------------------
   Application Name: "DIMG"
Application Version: 1.0.0
          File Name: Dimg.cpp
         Start Date: 27/06/2008
          Author(s): David Goodwin
          Copyright: (C) Copyright David Goodwin, 2008, 2009
            License: GNU General Public License
   File Description: DIMG Class Implementation
-----------------------------------------------------
	This file is part of "DIMG".

    This is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This software is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this software; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

	File Notes:
	------------------
*/



// ------------------
// Header files:
// ------------------
#include "Dimg.h"

// ------------------
// Constructor for download_manager class
Dimg::Dimg(QApplication *app) : QMainWindow(0)
{
        qDebug() << "Dimg Constructor...";

        // Remember the pointer to the application
	pApp = app;

        file = 0;
        fs = 0;
        img = 0;
        imgr = 0;

        imageDirty = false;
        imgOpen = false;

	// Then setup the Ui
	ui.setupUi(this);

        // Connect up a bunch of actions
        connect(ui.action_Open,SIGNAL(triggered()),this,SLOT(loadFile()));
        connect(ui.actionExtract_File,SIGNAL(triggered()),this,SLOT(extractFile()));
        connect(ui.actionFile_Properties,SIGNAL(triggered()),this,SLOT(showFileProperties()));
        connect(ui.actionE_xit,SIGNAL(triggered()),this,SLOT(close()));
        connect(ui.action_Close_Image,SIGNAL(triggered()),this,SLOT(closeImage()));
        connect(ui.action_About,SIGNAL(triggered()),this,SLOT(showAbout()));
        connect(ui.action_Read_Disk,SIGNAL(triggered()),this,SLOT(imageCurrentDrive()));
        connect(ui.actionWrite_Disk,SIGNAL(triggered()),this,SLOT(writeCurrentDrive()));
        connect(ui.action_Save,SIGNAL(triggered()),this,SLOT(saveCurrentImage()));
        connect(ui.actionSave_As,SIGNAL(triggered()),this,SLOT(saveCurrentImageAs()));

        // tree widget
        connect(ui.treeWidget,SIGNAL(collapsed(QModelIndex)),this,SLOT(resize_columns(QModelIndex)));
        connect(ui.treeWidget,SIGNAL(expanded(QModelIndex)),this,SLOT(resize_columns(QModelIndex)));

        // ----------------
        // Load the Filesystem Plugin
        loadFSPlugin();

        // ----------------
        // Prepare the Imager
        if (!imgr) {
            imgr = new Imager(this);
            imgr->set_parent_widget(this);
        }

        // Load the disk drivers
        if (imgr->load_drivers() == errors::E_SUCCESS) {
            qDebug() << "Preparing drive selector...";
            // ----------------
            // Add the floppy drives to the Disk menu
            QStringList fdns = imgr->getDescriptiveDriveNames();
            QStringList fds = imgr->getDriveNames();
            QPointer<QMenu> diskMenu = ui.menu_Select_Disk;
            QMenu targetMenu;
            targetMenu.setTitle("Select Drive");
            QPointer<QToolBar> disksel = ui.tb_DiskSelect;

            for (int i = 0; i < fds.count(); i++) {
                    QString title = "Use drive " + fdns.at(i);
                    QVariant drive = fds.at(i);
                    QAction *tempAction = new QAction(title, this);
                    tempAction->setData(drive);
                    tempAction->setCheckable(true);
                    disksel->addAction(tempAction);
                    diskMenu->addAction(tempAction);
                    diskSelActions << tempAction;
                    qDebug() << "Added Drive Action: " << title;
            }
            connect(diskMenu,SIGNAL(triggered(QAction *)),this,SLOT(diskSelect(QAction *)));
            connect(disksel,SIGNAL(actionTriggered(QAction *)),this,SLOT(diskSelect(QAction *)));

            if (imgr->configurable()) {
                qDebug() << "Driver is configurable. Adding configure option";
                QAction *dconf = new QAction("Configure Driver",this);

                ui.menu_Disk->addSeparator();
                ui.menu_Disk->addAction(dconf);
                connect(dconf,SIGNAL(triggered()),imgr,SLOT(configure()));
            }
        }
        else {
            // Disable features that rely on a DQPlugin if one couldnt be loaded.
            ui.action_Read_Disk->setEnabled(false);
            ui.actionWrite_Disk->setEnabled(false);
            ui.action_Format_Disk->setEnabled(false);
        }

        // Set the window title
        setProgramWindowTitle("");

        // Prepare the Filesystem Treeview widget
        prepareFSTreeView();

	// Show the user interface.
	show();
}

Dimg::~Dimg() {
    qDebug() << "Dimg Destructor...";
    if (fs) {
        fs->close();
        fs->exiting();
        delete fs;
        fs = 0;
    }
    if (file) {
        file->close();
        delete file;
        file = 0;
    }
    if (img) {
        img->close();
        delete img;
        img = 0;
    }
}

bool Dimg::loadFSPlugin() {
    qDebug() << "Loading FS Plugin...";
    QString plugin_file, alt_file;

    // Set the directory for the plugin to be the applications directory
    plugin_file = alt_file = QApplication::applicationDirPath();

    // Build the base filename
#ifdef Q_OS_WIN32
    plugin_file += "/fat12fs";
    alt_file += "/fsplugin";
#else
    plugin_file += "/libfat12fs";
    alt_file += "/libfsplugin";
#endif

    // If it is a debug build, use the debug build of the plugin
#ifdef _DIMG_DEBUG
    plugin_file += "d";
    alt_file += "d";
#endif

    // Add on the extension
#ifdef Q_OS_WIN32
    plugin_file += ".dll";
    alt_file += ".dll";
#else
    plugin_file += ".so";
    alt_file += ".so";
#endif

    QFileInfo info(plugin_file);

    if (!info.exists()) {
        plugin_file = alt_file;
        info.setFile(plugin_file);
        if (!info.exists()) {
            plugin_file = QFileDialog::getOpenFileName(this,"Locate Filesystem Plugin");
            if (plugin_file.isEmpty()) {
                QMessageBox::critical(0,"Plugin Load Error","No filesystem plugin specified. Exiting.");
            }
        }
    }

    qDebug() << "Creating plugin loader for file " << plugin_file;
    QPluginLoader *loader = new QPluginLoader(plugin_file);
    qDebug() << "Loading plugin...";
    if (loader->load()) {
        fs = qobject_cast<FSPlugin *>(loader->instance());
        if (!fs) {
            QMessageBox::critical(0,"Plugin Load Error","Loaded plugin was not of type FSPlugin.");
        }
    }
    else {
        QMessageBox::critical(0,"Plugin Load Error","A filesystem plugin could be found but not loaded.\nThis is likely a problem with the plugin.\nThe Plugin filename is:\n" + plugin_file);
    }
    delete loader;
    loader = 0;
    return (fs);
}

void Dimg::prepareFSTreeView() {
    qDebug() << "Preparing Filesystem Tree View...";
    // If there is no filesystem plugin loaded, disable the tree widget and
    // return.
    if (!fs) {
        ui.treeWidget->setDisabled(true);
        return;
    }

    // Otherwise, setup the column labels. This is where we would add any
    // filesystem-specific column headers.

    // Build the column labels
    QStringList cl;
    cl << "Filename";
    cl << "Attributes";
    cl << "Size";
    cl << "Date Modified";

    ui.treeWidget->setColumnCount(cl.size());
    ui.treeWidget->setHeaderLabels(cl);

    // Setup context menu
    ui.treeWidget->addAction(ui.actionExtract_File);
    ui.treeWidget->addAction(ui.actionFile_Properties);
    ui.treeWidget->setContextMenuPolicy(Qt::ActionsContextMenu);
}

void Dimg::diskSelect(QAction * dskAction) {
        qDebug() << "Disk Selected: " << dskAction->data();
	for (int i = 0; i < diskSelActions.count(); i++)
		diskSelActions.at(i)->setChecked(false);
	dskAction->setChecked(true);

        if (imgr)
            imgr->set_current_drive(dskAction->data().toString());
}

void Dimg::loadFile() {
    qDebug() << "Load file: prompting user for filename...";
    QString fn = QFileDialog::getOpenFileName(this,"Open Disk Image");
    if (fn.isEmpty()) return;
    loadFile(fn);
}

void Dimg::loadFile(QString filename) {
    qDebug() << "Attempting to open file: " << filename;

    qDebug() << "Opening file " << filename;
    if (file) {
        delete file;
        file = 0;
    }
    file = new QFile(filename);
    this->filename = filename;
    this->imageDirty = false;
    this->imgOpen = true;

    if (!file->open(QIODevice::ReadOnly)) {
        // Something has gone wrong
        QMessageBox::critical(0,"status","Unable to open file");
        return;
    }

    // If the file size is less than 5MB, load it all into memory.
    if (file->size() < 5242880) {
        QProgressDialog qpd;
        qpd.setWindowTitle("Loading...");
        qpd.setMaximum(file->size());
        if (img) delete img;
        img = new QBuffer();
        img->open(QIODevice::ReadWrite);

        int blocks = file->size() / 512;

        for (int i = 0; i < blocks; i++) {
            if (qpd.wasCanceled()) {
                file->close();
                delete file;
                file = 0;
                delete img;
                img = 0;
                return;
            }
            //qDebug() << "Reading block " << i;
            file->seek(i*512);
            img->seek(i*512);
            img->write(file->read(512));
            qpd.setValue(i);
        }
        qpd.setValue(file->size());
    }

    // Load the filesystem
    loadFileSystem();

    setProgramWindowTitle(filename);
    ui.action_Close_Image->setEnabled(true);
    ui.action_Image_Information->setEnabled(true);
    ui.action_Save->setEnabled(true);
    ui.actionSave_As->setEnabled(true);
}

void Dimg::setProgramWindowTitle() {
    qDebug() << "Update Program Window Title with current filename...";
    setProgramWindowTitle(this->filename);
}

void Dimg::setProgramWindowTitle(QString filename) {
    qDebug() << "Set program window title with filename: " << filename;
    QString appnv = version::application_name + " version " + utils::versionString();
    QString filename_out;

    filename_out = QDir::toNativeSeparators(filename);
    if (this->imageDirty)
        filename_out = filename_out + "*";

    if (this->imgOpen)
        if (!filename.isEmpty())
            setWindowTitle(appnv + " - [" + filename_out + "]");
        else
            setWindowTitle(appnv + " - [untitled.img*]");
    else
        setWindowTitle(appnv);
}

void Dimg::loadFSTree(QTreeWidgetItem *parent, QString directory) {
    qDebug() << "Load FS Tree...";
    QList<FSEntry *> entries = fs->getEntryList(directory);
    FSPlugin::Features f = fs->getFeatures();
    qDebug() << "Entries: " << entries.size();
    for (int i = 0; i < entries.size(); i++) {
        FSEntry *e = entries.at(i);

        // Time to build the strings for the entry:
        // Filename - Attributes (rhasd) - size
        QStringList qsl;
        qsl << e->fileName();

        FSPlugin::Attributes a = e->getAttributes();
        QString attributes;

        // Read-Only:
        if (a.testFlag(FSPlugin::A_ReadOnly))
            attributes.append("r");
        else
            attributes.append("-");

        // Hidden:
        if (a.testFlag(FSPlugin::A_Hidden))
            attributes.append("h");
        else
            attributes.append("-");

        // Archive:
        if (a.testFlag(FSPlugin::A_Archive))
            attributes.append("a");
        else
            attributes.append("-");

        // System File:
        if (a.testFlag(FSPlugin::A_System))
            attributes.append("s");
        else
            attributes.append("-");

        // Directory:
        if (a.testFlag(FSPlugin::A_Directory))
            attributes.append("d");
        else
            attributes.append("-");

        qsl << attributes;

        if (!a.testFlag(FSPlugin::A_Directory)) {
            // Compute the size
            quint64 sz = e->getSize();
            QString unit = " B";

            // Are we kilobytes?
            if (sz > 1024) {
                sz /= 1024;
                unit = " KB";
            }

            // Are we megabytes?
            if (sz > 1024) {
                sz /= 1024;
                unit = " MB";
            }

            // are we gigabytes? (thats one huge floppy disk)
            if (sz > 1024) {
                sz /= 1024;
                unit = " GB";
            }

            qsl << QString::number(sz,10) + unit;
        } else { qsl << ""; }

        // Check to see if modification date and time are supported.
        if (f.testFlag(FSPlugin::F_ModDate) &&
            f.testFlag(FSPlugin::F_ModTime)) {
            QDate d = e->getDate(FSEntry::D_ModifiedDate);
            QTime t = e->getTime(FSEntry::D_ModifiedTime);
            QString s = d.toString() + " " + t.toString();
            qsl << s;
        }
        // Otherwise, see if at least creation date and time are
        // supported.
        else if (f.testFlag(FSPlugin::F_CreationDate) &&
                 f.testFlag(FSPlugin::F_CreationTime)) {
            QDate d = e->getDate(FSEntry::D_CreatedDate);
            QTime t = e->getTime(FSEntry::D_CreatedTime);
            QString s = d.toString() + t.toString();
            qsl << s;
        }
        // If creation date and time arent supported either, we will just
        // ignore the date and time
        else {
            qsl << "";
        }


        QTreeWidgetItem *qtwi;
        if (parent == 0) {
            qtwi = new QTreeWidgetItem(ui.treeWidget, qsl,0);
        } else {
            qtwi = new QTreeWidgetItem(parent,qsl,0);
        }

        qtwi->setIcon(0,e->getIcon());
        qtwi->setData(0,Qt::UserRole, e->filePath());

        if (e->isDirectory()) {
            if (e->fileName() != "." && e->fileName() != "..")
                loadFSTree(qtwi,directory + e->fileName() + "/");
        }
    }

    // Now delete all of the FSEntrys:
    while (!entries.isEmpty())
        delete entries.takeFirst();

    ui.treeWidget->resizeColumnToContents(0);
    ui.treeWidget->setColumnWidth(0,ui.treeWidget->columnWidth(0) + 10);
    ui.treeWidget->resizeColumnToContents(1);
    ui.treeWidget->setColumnWidth(1,ui.treeWidget->columnWidth(1) + 10);
    ui.treeWidget->resizeColumnToContents(2);
    ui.treeWidget->setColumnWidth(2,ui.treeWidget->columnWidth(2) + 10);
    ui.treeWidget->resizeColumnToContents(3);
    ui.treeWidget->setColumnWidth(3,ui.treeWidget->columnWidth(3) + 10);
}

void Dimg::resize_columns(const QModelIndex &index) {
    qDebug() << "Resizing columns...";
    ui.treeWidget->resizeColumnToContents(0);
    ui.treeWidget->setColumnWidth(0,ui.treeWidget->columnWidth(0) + 10);
    ui.treeWidget->resizeColumnToContents(1);
    ui.treeWidget->setColumnWidth(1,ui.treeWidget->columnWidth(1) + 10);
    ui.treeWidget->resizeColumnToContents(2);
    ui.treeWidget->setColumnWidth(2,ui.treeWidget->columnWidth(2) + 10);
    ui.treeWidget->resizeColumnToContents(3);
    ui.treeWidget->setColumnWidth(3,ui.treeWidget->columnWidth(3) + 10);
}

QString Dimg::getSelectedFile() {
    qDebug() << "Getting selected file...";
    // Get selected filename - Qt::UserRole

    QTreeWidgetItem *qtwi = ui.treeWidget->currentItem();
    if (qtwi == 0) return "";

    QString d = qtwi->data(0,Qt::UserRole).toString();
    return d;
}

void Dimg::extractFile() {
    qDebug() << "Extracting file...";
    QString fn = getSelectedFile();
    if (fn.isEmpty()) return;
    extractFile(fn);
}

void Dimg::showFileProperties() {
    qDebug() << "Showing File Properties...";
    QString fn = getSelectedFile();
    if (fn.isEmpty()) return;
    showFileProperties(fn);
}

void Dimg::extractFile(QString filename) {
    qDebug() << "Extracting file " << filename;
    QString dest_fn = QFileDialog::getSaveFileName(this,"Extract File...",filename);
    fs->copyFile(filename,dest_fn);
}

void Dimg::showFileProperties(QString filename) {
    qDebug() << "Showing file properties for " << filename;
    FileProperties fp(this);
    FSEntry *e = fs->getEntryByName(filename);
    fp.setFileEntry(e);
    fp.exec();
    
}

void Dimg::closeEvent(QCloseEvent *event) {
    qDebug() << "Close event: application exiting.";

    // If there is an unsaved image open, ask if it should be saved.
    if (prompt_image_save() == errors::E_USER_CANCELED) {
        if (event)
            event->ignore();
        return;
    }

    closeImage();
    // Perform exit stuff for the filesystem plugin
    if (fs) {
        fs->exiting();
        delete fs;
        fs = 0;
    }

    // Close any open files
    if (file) {
        delete file;
        file = 0;
    }

    if (img) {
        delete img;
        img = 0;
    }

    // Exit
    pApp->exit(0);
}

void Dimg::closeImage() {
    qDebug() << "Closing current image...";
    if (fs) {
        fs->close();
        ui.treeWidget->clear();
    }
    if (img) {
        img->close();
        delete img;
        img = 0;
    }
    if (file) {
        file->close();
        delete file;
        file = 0;
    }
    ui.action_Close_Image->setEnabled(false);
    ui.actionExtract_File->setEnabled(false);
    ui.actionFile_Properties->setEnabled(false);
    ui.action_Image_Information->setEnabled(false);
    ui.action_Save->setEnabled(false);
    ui.actionSave_As->setEnabled(false);
    imageDirty = false;
    filename = "";
    imgOpen = false;
    setProgramWindowTitle();

}

void Dimg::createImage(){
    qDebug() << "Creating blank image...";
    img = new QBuffer();
    imageDirty = true;
    imgOpen = true;
    filename = "";
    ui.action_Close_Image->setEnabled(true);
    ui.action_Image_Information->setEnabled(true);
    ui.action_Save->setEnabled(true);
    ui.actionSave_As->setEnabled(true);
    setProgramWindowTitle();
}

void Dimg::loadFileSystem(){
    qDebug() << "Loading filesystem...";
    if (fs)
        if (!fs->isClosed()) {
            qDebug() << "Closing previous filesystem instance...";
            fs->close();
        }

    qDebug() << "Loading filesystem...";
    if (fs) {
        if (img) {
            qDebug() << "Loading filesystem from buffer";
            fs->load(img);
        }
        else {
            qDebug() << "Loading filesystem from file";
            fs->load(file);
        }
    }

    qDebug() << "Displaying File Tree...";
    ui.treeWidget->clear();
    if (fs) {
        loadFSTree(0,"/");
        ui.actionExtract_File->setEnabled(true);
        ui.actionFile_Properties->setEnabled(true);
    }

    FSPlugin::Features f = fs->getFeatures();

    if (f.testFlag(FSPlugin::F_VolumeLabel)) {
        qDebug() << "Displaying Volume Label...";
        ui.statusBar->showMessage("Volume Label: " + fs->volumeLabel());
    }
}

void Dimg::showAbout() {
    qDebug() << "Showing about screen...";
    QString fsPluginName = "None Loaded", fsPluginVersion;
    QString dqPluginName = "None Loaded", dqPluginVersion;


    AboutDialog ad;
    if (fs) {
        fsPluginName = fs->getPluginName();
        fsPluginVersion = fs->getVersionString();
    }
    if (imgr->driver_ready()) {
        dqPluginName = imgr->get_driver_name();//dqdrv->getPluginName();
        dqPluginVersion = imgr->get_driver_version();//dqdrv->getVersionString();
    }

    ad.setData(fsPluginName, fsPluginVersion, dqPluginName, dqPluginVersion);
    ad.exec();
}

int Dimg::saveCurrentImage() {
    if (this->filename.isEmpty()) {
        // No filename is set its never been saved before. Force Save-As.
        return saveCurrentImageAs();
    }

    QFile outfile(this->filename);
    outfile.open(QIODevice::ReadWrite);
    outfile.write(img->buffer());
    outfile.flush();
    outfile.close();

    imageDirty = false;         // The file has been saved
    setProgramWindowTitle();    // Update the window title
    return errors::E_SUCCESS;   // Return success
}

int Dimg::saveCurrentImageAs() {
    QString nfn = QFileDialog::getSaveFileName(this,"Save Disk Image As",this->filename,"All Files (*.*)");
    if (!nfn.isEmpty()) {
        this->filename = nfn;       // Change the filename
        setProgramWindowTitle();    // Update the title
        return saveCurrentImage();  // Save the image
    } else {
        return errors::E_USER_CANCELED;
    }
}

int Dimg::prompt_image_save() {
    // Is there an image open? Is it dirty? If so, save it.
    if (imgOpen && imageDirty) {
        qDebug() << "Unsaved image open - prompting to save...";
        QMessageBox msgbox;
        msgbox.setText("The current image has been modified. Do you want to save before closing it?");
        msgbox.setStandardButtons(QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel);
        msgbox.setDefaultButton(QMessageBox::Save);
        msgbox.setIcon(QMessageBox::Question);
        int ret = msgbox.exec();

        switch (ret) {
            case (QMessageBox::Save):
                qDebug() << "-> User chose to save.";
                saveCurrentImage();
                break;
            case (QMessageBox::Cancel):
                qDebug() << "-> User chose to cancel.";
                return errors::E_USER_CANCELED;
                break;
            case (QMessageBox::Discard):
                qDebug() << "-> User chose to discard.";
                break;
            default:
                break;
        }
    }
    return errors::E_SUCCESS;
}

void Dimg::imageCurrentDrive(){
    qDebug() << "Imaging current drive...";
    /* Ok, here is what we need to do:
        1. If there is an image open, close it.
        1.1. If the image is dirty, ask to save it.
        1.2. Clear buffers, close filesystem, etc.
        2. Prepare a QBuffer to image into
        3. Tell The Imager to Get Imaging!
     */

    // First, we cant image if the Imagers driver isn't ready.
    if (!imgr->driver_ready()) {
        qDebug() << "CRITICAL: Unable to image - Imager Driver not ready.";
        QMessageBox::critical(0,"Unable to Create Image", "No drive access plugin is currently active. Without a drive access plugin\n" + version::application_name + " can not read or write images to drives.");
        return;
    }

    // Ask if we should save the image
    if (prompt_image_save() == errors::E_USER_CANCELED)
        return;

    // Ok, the current image has been dealt with. Close it.
    closeImage();
    qDebug() << "Image Closed.";

    // Now we need to prepare something to image the disk into
    createImage();

    // And lastly, we need to actually image the disk in the current drive.
    int errcode = imgr->read_image(img);

    if (errcode) {
        // We do not display an error here as The Imager should have already
        // done it. Just close the image and move on.
        closeImage();
    } else {
        qDebug() << "Imaging Successful";

        // Open the image
        img->open(QIODevice::ReadWrite);

        // Load the Filesystem
        loadFileSystem();

        // done!
        QMessageBox::information(0,"Complete.", "Disk Imaging Complete.");
    }
}

void Dimg::writeCurrentDrive() {
    qDebug() << "Writing current drive...";
    int errcode = imgr->write_image(img);
    if (!errcode) {
        qDebug() << "Imaging Successful";

        // Open the image
        img->open(QIODevice::ReadWrite);

        // done!
        QMessageBox::information(0,"Complete.", "Disk Imaging Complete.");
    }
}
